use napi::bindgen_prelude::{FromNapiValue, ToNapiValue};
use napi_derive::napi;

#[napi(string_enum)]
pub enum DiffType {
  Insert,
  Delete,
  Equal,
}

#[napi(object)]
pub struct DiffChunk {
  pub diff_type: DiffType,
  pub text: String,
}

#[napi]
pub struct DiffClient {}

#[napi]
impl DiffClient {
  #[napi(constructor)]
  pub fn new() -> DiffClient {
    DiffClient {}
  }

  #[napi]
  pub fn diff(&self, _text1: String, _text2: String) -> Vec<DiffChunk> {
    let diff = dissimilar::diff(_text1.as_str(), _text2.as_str());

    let mut result = vec![];
    for d in diff {
      match d {
        dissimilar::Chunk::Equal(_text) => {
          result.push(DiffChunk {
            diff_type: DiffType::Equal,
            text: "".to_string(),
          });
        }
        dissimilar::Chunk::Delete(text) => {
          result.push(DiffChunk {
            diff_type: DiffType::Delete,
            text: text.to_string(),
          });
        }
        dissimilar::Chunk::Insert(text) => {
          result.push(DiffChunk {
            diff_type: DiffType::Insert,
            text: text.to_string(),
          });
        }
      }
    }

    return result;
  }

  /// use https://docs.rs/diffmatchpatch/latest/diffmatchpatch/struct.DiffMatchPatch.html#method.diff_lines_to_chars
  /// then diff the chars.
  /// then convert back to lines.
  ///
  /// takes in two strings. splits based on newlines.
  /// returns diffs based on lines.
  #[napi]
  pub fn diff_lines(&self, _text1: String, _text2: String) -> Vec<DiffChunk> {
    let diff = dissimilar::diff(_text1.as_str(), _text2.as_str());

    let mut result = vec![];
    for d in diff {
      match d {
        dissimilar::Chunk::Equal(_text) => {
          result.push(DiffChunk {
            diff_type: DiffType::Equal,
            text: "".to_string(),
          });
        }
        dissimilar::Chunk::Delete(text) => {
          result.push(DiffChunk {
            diff_type: DiffType::Delete,
            text: text.to_string(),
          });
        }
        dissimilar::Chunk::Insert(text) => {
          result.push(DiffChunk {
            diff_type: DiffType::Insert,
            text: text.to_string(),
          });
        }
      }
    }

    return result;
  }
}
