// what methods do i want to support here.
// 1. isInBadDir
// 2. isBadFile
// 3. vscode.workspace.asRelativePath
// 4. vscode.fs.stat

use anyhow::Error;
use encoding_rs::UTF_8;
use log::info;
use std::path::Path;
use tokio::fs;

pub fn is_in_bad_dir(file_path: &Path) -> Result<bool, Error> {
  let item_path = file_path
    .to_str()
    .ok_or(anyhow::anyhow!("Failed to convert path to string"))?;
  let is_bad_dir =
    item_path.contains("node_modules") || item_path.contains(".git");
  Ok(is_bad_dir)
}

pub fn is_good_file(file_path: &Path) -> Result<(), Error> {
  let item_path = file_path
    .to_str()
    .ok_or(anyhow::anyhow!("Failed to convert path to string"))?;

  let path = Path::new(item_path);
  let file_name = path
    .file_name()
    .ok_or(anyhow::anyhow!("Failed to get file name"))?
    .to_str()
    .ok_or(anyhow::anyhow!("Failed to convert file name to string"))?;

  match file_name {
    "package-lock.json" | "pnpm-lock.yaml" | "yarn.lock" | "composer.lock"
    | "Gemfile.lock" | "bun.lockb" => {
      return Err(anyhow::anyhow!("File is just a lock file"));
    }
    _ => {}
  }

  if file_name.starts_with(".env") {
    return Err(anyhow::anyhow!("File is an env file"));
  }

  let components: Vec<_> = path
    .components()
    .map(|comp| comp.as_os_str().to_str().unwrap_or(""))
    .collect();
  if components.contains(&".git")
    || components.contains(&".svn")
    || components.contains(&".hg")
  {
    return Err(anyhow::anyhow!("File is in a version control directory"));
  }
  let extension_result = path.extension();

  return match extension_result {
    Some(extension) => {
      let extension_str = extension
        .to_str()
        .ok_or(anyhow::anyhow!("Failed to convert extension to string"))?;
      let bad_extensions = vec![
        "lock", "bak", "tmp", "bin", "exe", "dll", "so", "lockb", "qwoff",
        "isl", "csv", "pdf", // add ms word, excel, powerpoint, etc.
        "doc", "docx", "xls", "xlsx", "ppt", "pptx", "odt", "ods", "odp",
        "odg", "odf", "sxw", "sxc", "sxi", "sxd", "sdc", // add images
        "jpg", "jpeg", "png", "gif", "bmp", "tif", // add audio
        "mp3", "wav", "wma", "ogg", "flac", "aac", // add video
        "mp4", "mov", "wmv", "flv", "avi", // add archives
        "zip", "tar", "gz", "7z", "rar", "tgz", "dmg", "iso", "cue", "mdf",
        "mds", "vcd", "toast", "img", "apk", "msi", "cab", "tar.gz", "tar.xz",
        "tar.bz2", "tar.lzma", "tar.Z", "tar.sz", "lzma", // add fonts
        "ttf", "otf", "woff", "woff2", "eot", "webp", "vsix", "rmeta", "rlib",
      ];
      match bad_extensions.contains(&extension_str) {
        true => {
          info!(
            "File has bad extension: {} for {:?}",
            extension_str, file_name
          );
          return Err(anyhow::anyhow!("File is just a lock file"));
        }
        _ => Ok(()),
      }
    }
    None => {
      info!("Failed to get extension for: {}", file_name);
      Ok(())
    }
  };
}

// use binaryornot::is_binary;
// use anyhow::Context;
// implement the buffer above:
pub async fn is_good_file_runtime_check(
  file_path: &Path,
  // _buffer: &[u8],
) -> Result<(), Error> {
  match get_file_size(file_path).await {
    Ok(size) if size > 2 * 1024 * 1024 => {
      return Err(anyhow::anyhow!("Buffer is too large"));
    }
    Err(e) => return Err(e),
    _ => {}
  }

  // if is_binary(file_path).context("Failed to check if file is binary")? {
  // 	return Err(anyhow::anyhow!("File is binary"));
  // }

  Ok(())
}

pub async fn read_string_without_bom(
  file_path: &Path,
) -> Result<String, Error> {
  let file_buffer = match fs::read(file_path).await {
    Ok(buffer) => buffer,
    Err(e) => {
      return Err(anyhow::anyhow!(
        "Failed to read file buffer: {}",
        e.to_string()
      ))
    }
  };

  let (cow, _) = UTF_8.decode_with_bom_removal(&file_buffer);

  Ok(cow.to_string())
}

pub fn as_relative_path(
  base_path: &Path,
  file_path: &Path,
) -> Result<String, Error> {
  let relative_path = file_path.strip_prefix(base_path)?;
  Ok(
    relative_path
      .to_str()
      .ok_or(anyhow::anyhow!("Failed to convert relative path to string"))?
      .to_string(),
  )
}

pub async fn get_file_size(file_path: &Path) -> Result<u64, Error> {
  let metadata = fs::metadata(file_path).await?;

  Ok(metadata.len())
}

#[cfg(test)]
mod tests {
  use super::*;
  use std::path::Path;
  use tokio::io::AsyncWriteExt;

  #[test]
  fn test_is_in_bad_dir() {
    let path = Path::new("src/node_modules/test.rs");
    assert_eq!(is_in_bad_dir(&path).unwrap(), true);

    let path = Path::new("src/.git/test.rs");
    assert_eq!(is_in_bad_dir(&path).unwrap(), true);

    let path = Path::new("src/test.rs");
    assert_eq!(is_in_bad_dir(&path).unwrap(), false);
  }

  #[test]
  fn test_is_good_file() {
    let path = Path::new("src/test.rs");
    assert_eq!(is_good_file(&path).is_ok(), true);

    let path = Path::new("src/test.exe");
    assert_eq!(is_good_file(&path).is_err(), true);
  }

  #[tokio::test]
  async fn test_is_good_file_runtime_check() {
    let temp_dir = tempfile::tempdir().unwrap();
    let temp_file_path = temp_dir.path().join("test_file");
    let mut temp_file = fs::File::create(&temp_file_path).await.unwrap();
    temp_file.write_all(b"Hello, world!").await.unwrap();
    let _buffer = fs::read(&temp_file_path).await.unwrap();
    assert_eq!(
      is_good_file_runtime_check(&temp_file_path).await.is_ok(),
      true
    );
    temp_dir.close().unwrap();

    // let temp_dir = tempfile::tempdir().unwrap();
    // let temp_file_path = temp_dir.path().join("test_file");
    // let mut temp_file = fs::File::create(&temp_file_path).await.unwrap();
    // temp_file.write_all(&[0, 159, 146, 150]).await.unwrap(); // Invalid UTF-8 sequence
    // let buffer = fs::read(&temp_file_path).await.unwrap();
    // assert_eq!(
    //   is_good_file_runtime_check(&temp_file_path).await.is_err(),
    //   true
    // );
    // temp_dir.close().unwrap();
  }

  #[tokio::test]
  async fn test_bom_file() {
    const BOM: [u8; 3] = [0xEF, 0xBB, 0xBF];
    const CONTENT: &str = "Hello, world!";

    // Write this to a temp file
    let temp_dir = tempfile::tempdir().unwrap();
    let temp_file_path = temp_dir.path().join("test_file");
    let mut temp_file = fs::File::create(&temp_file_path).await.unwrap();
    temp_file.write_all(&BOM).await.unwrap();
    temp_file.write_all(CONTENT.as_bytes()).await.unwrap();

    // expect that we read the file with tokio as the CONTENT
    let file_contents = read_string_without_bom(&temp_file_path).await.unwrap();

    // Check string equality of CONTENT (&str) to file_contents (String)
    assert_eq!(CONTENT, file_contents);
  }

  #[test]
  fn test_as_relative_path() {
    let base_path = Path::new("/home/user/src");
    let file_path = Path::new("/home/user/src/test.rs");
    assert_eq!(as_relative_path(&base_path, &file_path).unwrap(), "test.rs");

    let file_path = Path::new("/home/user/test.rs");
    assert!(as_relative_path(&base_path, &file_path).is_err());
  }
}
