use std::collections::HashSet;
use std::path::MAIN_SEPARATOR_STR;
use std::process::Command;

pub fn list_ignored_files_and_directories(
  workspace_root_path: &str,
  should_return_absolute_paths: bool,
) -> Result<HashSet<String>, Box<dyn std::error::Error>> {
  let mut gitignored_files = HashSet::new();

  let commands = vec![
    vec![
      "git",
      "ls-files",
      "--others",
      "--ignored",
      "--exclude-standard",
      "--directory",
      "--no-empty-directory"
    ],
    // FIXME(sualeh): this is super sketchy and might totally break in like a bazillion ways. i dont like it.
    vec![
      "git submodule foreach --quiet 'git -C $toplevel/$path ls-files --others --ignored --exclude-standard --directory --no-empty-directory | (while read line; do echo $path/$line; done)'",
    ],
  ];

  for command in commands {
    let output = Command::new(&command[0])
      .args(&command[1..])
      .current_dir(workspace_root_path)
      .output()?;

    if output.status.success() {
      let files = String::from_utf8(output.stdout)?
        .lines()
        .filter(|line| !line.is_empty())
        .map(|line| {
          let line = line.replace("/", MAIN_SEPARATOR_STR);

          if should_return_absolute_paths {
            let mut path = std::path::PathBuf::from(workspace_root_path);

            path.push(line.clone());

            match path.canonicalize() {
              Ok(canonical_path) => {
                canonical_path.to_string_lossy().into_owned()
              }
              Err(_) => String::from(line),
            }
          } else {
            String::from(line)
          }
        })
        .collect::<HashSet<_>>();

      gitignored_files.extend(files);
    }
  }

  Ok(gitignored_files)
}

pub fn list_ignored_files_with_absolute_paths(
  workspace_root_path: &str,
) -> Result<HashSet<String>, Box<dyn std::error::Error>> {
  let mut gitignored_files = HashSet::new();

  let commands = vec![
    vec![
      "git",
      "ls-files",
      "--others",
      "--ignored",
      "--exclude-standard",
    ],
    // FIXME(sualeh): this is super sketchy and might totally break in like a bazillion ways. i dont like it.
    vec![
      "sh",
      "-c",
      "git submodule foreach --quiet 'git ls-files --others --ignored --exclude-standard | sed \"s|^|$path/|\"'",
    ],
  ];

  for command in commands {
    let output = Command::new(&command[0])
      .args(&command[1..])
      .current_dir(workspace_root_path)
      .output()?;

    if output.status.success() {
      let files = String::from_utf8(output.stdout)?
        .lines()
        .filter(|line| !line.is_empty())
        .map(|line| String::from(line))
        .collect::<HashSet<_>>();

      gitignored_files.extend(files);
    }
  }

  Ok(gitignored_files)
}

pub async fn is_git_ignored(
  workspace_root_path: &str,
  file_path: &str,
) -> Result<bool, anyhow::Error> {
  let output = tokio::process::Command::new("git")
    .args(&["check-ignore", file_path])
    .current_dir(workspace_root_path)
    .output()
    .await?;

  Ok(output.status.success())
}

pub async fn is_git_directory(
  workspace_root_path: &str,
) -> Result<bool, anyhow::Error> {
  let output = tokio::process::Command::new("git")
    .args(&["rev-parse", "--is-inside-work-tree"])
    .current_dir(workspace_root_path)
    .output()
    .await?;

  Ok(output.status.success())
}

#[cfg(test)]
mod tests {
  use super::*;
  use std::fs::File;
  use std::io::Write;

  #[test]
  fn test_no_ignored_files() {
    let dir = tempfile::tempdir().unwrap();
    let gitignored_files =
      list_ignored_files_and_directories(dir.path().to_str().unwrap(), false)
        .unwrap();
    Command::new("git")
      .args(&["init"])
      .current_dir(dir.path())
      .output()
      .unwrap();
    assert_eq!(gitignored_files.len(), 0);
  }

  #[test]
  fn test_one_ignored_file() {
    let dir = tempfile::tempdir().unwrap();
    println!("tempdir: {:?}", dir);
    let file_path = dir.path().join("ignored.txt");
    let mut file = File::create(&file_path).unwrap();
    writeln!(file, "This is an ignored file.").unwrap();

    let gitignore_path = dir.path().join(".gitignore");
    let mut gitignore = File::create(&gitignore_path).unwrap();
    writeln!(gitignore, "ignored.txt").unwrap();

    Command::new("git")
      .args(&["init"])
      .current_dir(dir.path())
      .output()
      .unwrap();
    let gitignored_files =
      list_ignored_files_and_directories(dir.path().to_str().unwrap(), false)
        .unwrap();
    println!(
      "ignored files for test_one_ignored_file: {:?}",
      gitignored_files
    );
    // assert_eq!(gitignored_files.len(), 1);
    assert!(gitignored_files.contains(&String::from("ignored.txt")));
  }

  #[test]
  fn test_multiple_ignored_files() {
    let dir = tempfile::tempdir().unwrap();
    println!("tempdir: {:?}", dir);
    let file_path1 = dir.path().join("ignored1.txt");
    let file_path2 = dir.path().join("ignored2.txt");
    let mut file1 = File::create(&file_path1).unwrap();
    let mut file2 = File::create(&file_path2).unwrap();
    writeln!(file1, "This is an ignored file.").unwrap();
    writeln!(file2, "This is another ignored file.").unwrap();

    let gitignore_path = dir.path().join(".gitignore");
    let mut gitignore = File::create(&gitignore_path).unwrap();
    writeln!(gitignore, "*.txt").unwrap();

    Command::new("git")
      .args(&["init"])
      .current_dir(dir.path())
      .output()
      .unwrap();
    let gitignored_files =
      list_ignored_files_and_directories(dir.path().to_str().unwrap(), false)
        .unwrap();
    println!(
      "ignored files for test_multiple_ignored_files: {:?}",
      gitignored_files
    );
    // assert_eq!(gitignored_files.len(), 2);
    assert!(gitignored_files.contains(&String::from("ignored1.txt")));
    assert!(gitignored_files.contains(&String::from("ignored2.txt")));
  }

  #[test]
  fn test_git_submodule_ignored_files() {
    let dir = tempfile::tempdir().unwrap();
    let submodule_path = dir.path().join("submodule");
    std::fs::create_dir(&submodule_path).unwrap();

    let o = Command::new("git")
      .args(&["init"])
      .current_dir(&submodule_path)
      .output()
      .unwrap();
    println!("git init output: {:?}", o);

    let file_path = submodule_path.join("ignored.txt");
    let mut file = File::create(&file_path).unwrap();
    writeln!(file, "This is an ignored file.").unwrap();

    let file2 = submodule_path.join("ignored2.txt");
    let mut file = File::create(&file2).unwrap();
    writeln!(file, "This is another ignored file.").unwrap();

    let gitignore_path = submodule_path.join(".gitignore");
    let mut gitignore = File::create(&gitignore_path).unwrap();
    writeln!(gitignore, "*.txt").unwrap();

    let o = Command::new("git")
      .args(&["init"])
      .current_dir(dir.path())
      .output()
      .unwrap();
    println!("git init output: {:?}", o);

    // make a commit in the submodule
    let o = Command::new("git")
      .args(&["add", "."])
      .current_dir(&submodule_path)
      .output()
      .unwrap();
    println!("git add output: {:?}", o);
    let o = Command::new("git")
      .args(&["commit", "-m", "initial commit"])
      .current_dir(&submodule_path)
      .output()
      .unwrap();
    println!("git commit output: {:?}", o);

    let o = Command::new("git")
      .args(&["submodule", "add", "./submodule"])
      .current_dir(dir.path())
      .output()
      .unwrap();
    println!("git submodule add output: {:?}", o);

    let gitignored_files =
      list_ignored_files_and_directories(dir.path().to_str().unwrap(), false)
        .unwrap();
    println!(
      "ignored files for test_git_submodule_ignored_files: {:?}",
      gitignored_files
    );
    assert!(gitignored_files.contains(&String::from("submodule/ignored.txt")));
    assert!(gitignored_files.contains(&String::from("submodule/ignored2.txt")));
  }

  #[test]
  fn test_multiple_ignored_files_in_current_dir() {
    let gitignored_files =
      list_ignored_files_and_directories(".", false).unwrap();
    assert!(gitignored_files.len() > 1);

    // print a sample of the ignored files
    let mut count = 0;
    for file in gitignored_files {
      println!("ignored file: {:?}", file);
      count += 1;
      if count > 10 {
        break;
      }
    }
  }

  #[tokio::test]
  async fn test_file_not_ignored() {
    let dir = tempfile::tempdir().unwrap();
    let file_path = dir.path().join("not_ignored.txt");
    let mut file = File::create(&file_path).unwrap();
    writeln!(file, "This is not an ignored file.").unwrap();

    Command::new("git")
      .args(&["init"])
      .current_dir(dir.path())
      .output()
      .unwrap();
    let is_ignored =
      is_git_ignored(dir.path().to_str().unwrap(), "not_ignored.txt")
        .await
        .unwrap();
    assert_eq!(is_ignored, false);
  }

  #[tokio::test]
  async fn test_file_ignored() {
    let dir = tempfile::tempdir().unwrap();
    let file_path = dir.path().join("ignored.txt");
    let mut file = File::create(&file_path).unwrap();
    writeln!(file, "This is an ignored file.").unwrap();

    let gitignore_path = dir.path().join(".gitignore");
    let mut gitignore = File::create(&gitignore_path).unwrap();
    writeln!(gitignore, "ignored.txt").unwrap();

    Command::new("git")
      .args(&["init"])
      .current_dir(dir.path())
      .output()
      .unwrap();
    let is_ignored =
      is_git_ignored(dir.path().to_str().unwrap(), "ignored.txt")
        .await
        .unwrap();
    assert_eq!(is_ignored, true);
  }

  #[tokio::test]
  async fn test_file_ignored_with_wildcard() {
    let dir = tempfile::tempdir().unwrap();
    let file_path = dir.path().join("ignored.txt");
    let mut file = File::create(&file_path).unwrap();
    writeln!(file, "This is an ignored file.").unwrap();

    let gitignore_path = dir.path().join(".gitignore");
    let mut gitignore = File::create(&gitignore_path).unwrap();
    writeln!(gitignore, "*.txt").unwrap();

    Command::new("git")
      .args(&["init"])
      .current_dir(dir.path())
      .output()
      .unwrap();
    let is_ignored =
      is_git_ignored(dir.path().to_str().unwrap(), "ignored.txt")
        .await
        .unwrap();
    assert_eq!(is_ignored, true);
  }
}
