#[cfg(test)]
mod tests {
  use super::super::*;

  use std::fs::File;
  use std::io::Write;
  use tempfile::tempdir;

  #[tokio::test]
  async fn test_empty_tree() {
    let tree = MerkleTree::empty_tree();
    assert_eq!(tree.root_path, "".to_string());
    assert_eq!(tree.files.len(), 0);
  }

  #[tokio::test]
  async fn test_empty_folder() {
    let temp_dir = tempdir().unwrap();
    let temp_dir_path = temp_dir.path().to_str().unwrap().to_string();

    // Test new() function
    match MerkleTree::new(Some(temp_dir_path.clone())).await {
      Ok(tree) => {
        assert_eq!(tree.root_path, temp_dir_path);
        assert_eq!(tree.files.len(), 1);
      }
      Err(_) => panic!("Failed to create new MerkleTree"),
    }
  }

  #[tokio::test]
  async fn test_local_construction() {
    let temp_dir = tempdir().unwrap();
    let temp_dir_path = temp_dir.path().to_str().unwrap().to_string();

    // Create a file in the temp directory
    let file_path = temp_dir_path.clone() + "/test.txt";
    let mut file = File::create(&file_path).unwrap();
    writeln!(file, "Hello, world!").unwrap();
    // print the contents of hte directory using ls
    // let path = Path::new(&temp_dir_path);

    // Test construct_merkle_tree() function
    let new_set = std::collections::HashSet::<String>::new();
    let tree =
      MerkleTree::construct_merkle_tree(temp_dir_path.clone(), new_set, false)
        .await;
    let mut tree = match tree {
      Ok(tree) => {
        assert_eq!(tree.files.len(), 2);
        assert!(tree.files.contains_key(&file_path));

        tree
      }
      Err(_) => panic!("Failed to construct MerkleTree"),
    };
    // Update the file
    writeln!(file, "Hello, again!").unwrap();
    match tree.update_file(file_path.clone()).await {
      Ok(_) => (),
      Err(_) => panic!("Failed to update file"),
    };
    // Check that the file's hash has been updated in the tree
    let file_node_option = tree.files.get(&file_path);
    match file_node_option {
      Some(file_node) => {
        let hash = file_node.node.read().await.hash.clone();
        assert_eq!(hash, compute_hash("Hello, world!\nHello, again!\n"));
      }
      None => panic!("Failed to get file node"),
    }
    // Test delete_file() function
    match tree.delete_file(file_path.clone()).await {
      Ok(_) => {
        assert!(!tree.files.contains_key(&file_path));
      }
      Err(_) => panic!("Failed to delete file"),
    }
  }

  #[tokio::test]
  async fn test_ignore_files() {
    let temp_dir = tempdir().unwrap();
    let temp_dir_path = temp_dir.path().to_str().unwrap().to_string();

    // Create a .cursorignore file in the temp directory
    let ignore_file_path = temp_dir_path.clone() + "/.cursorignore";
    let mut ignore_file = File::create(&ignore_file_path).unwrap();
    writeln!(ignore_file, "src/*\n!src/hi.txt").unwrap();

    // Create a src directory and a hi.txt file inside it
    let src_dir_path = temp_dir_path.clone() + "/src";
    fs::create_dir_all(&src_dir_path).unwrap();
    let hi_file_path = src_dir_path.clone() + "/hi.txt";
    let mut hi_file = File::create(&hi_file_path).unwrap();
    writeln!(hi_file, "Hello, world!").unwrap();

    // create a nothi.txt in the src directory
    let nothi_file_path = src_dir_path.clone() + "/nothi.txt";
    let mut nothi_file = File::create(&nothi_file_path).unwrap();
    writeln!(nothi_file, "Hello, world!").unwrap();

    // Create a README.md in the root directory
    let readme_file_path = temp_dir_path.clone() + "/README.md";
    let mut readme_file = File::create(&readme_file_path).unwrap();
    writeln!(readme_file, "# README").unwrap();

    // Construct the Merkle tree with ripgrep ignore
    let tree = MerkleTree::construct_merkle_tree_with_ripgrep_ignore(
      temp_dir_path.clone(),
    )
    .await;
    let tree = match tree {
      Ok(tree) => tree,
      Err(_) => panic!("Failed to construct MerkleTree"),
    };

    // Assert that the number of files is 3 (.cursorignore, hi.txt, readme.md)
    assert_eq!(tree.get_num_embeddable_files().await.unwrap(), 3);

    let all_files = tree.get_all_files().await.unwrap();
    assert_eq!(all_files.len(), 3);

    assert!(all_files.contains(&ignore_file_path));
    assert!(all_files.contains(&hi_file_path));
    assert!(all_files.contains(&readme_file_path));
  }

  #[tokio::test]
  #[ignore] // it is a local-only test
  async fn test_dump_merkle_tree_to_file() {
    // needs to have a .cursorignore with .github in it
    let file_path = "/Users/arvid/code/abseil-cpp".to_string();

    // Construct the Merkle tree
    // let tree =
    //   MerkleTree::construct_merkle_tree(file_path.clone(), new_set, false)
    //     .await;
    let tree =
      MerkleTree::construct_merkle_tree_with_ripgrep_ignore(file_path.clone())
        .await;
    let tree = match tree {
      Ok(tree) => tree,
      Err(_) => panic!("Failed to construct MerkleTree"),
    };

    println!("Finished constructing merkle tree.");

    // Print the tree using the fmt function
    let tree_string = format!("{}", tree);

    let mut file =
      File::create("/Users/arvid/code/merkle_output_test.txt").unwrap();
    match write!(file, "{}", tree_string) {
      Ok(_) => (),
      Err(_) => panic!("Failed to write the MerkleTree to file"),
    };

    // Compare the tree_string against the reference file in merkle_output.txt
    let reference_file_content = include_str!("merkle_output.txt");
    assert_eq!(tree_string, reference_file_content);

    // let mut file = File::create("/Users/arvid/code/merkle_output.txt").unwrap();
    // match write!(file, "{}", tree_string) {
    //   Ok(_) => (),
    //   Err(_) => panic!("Failed to write the MerkleTree to file"),
    // };
  }

  //   // TEST: manual sanity test: go to the git root and then index construct a merle tree on the whole repo.
  //   #[tokio::test]
  //   async fn test_git_root_merkle_tree() {

  //   use std::process::Command;
  //     // Get the git root path
  //     let output = Command::new("git")
  //       .args(&["rev-parse", "--show-toplevel"])
  //       .output()
  //       .expect("Failed to get git root path");

  //     let git_root_path = String::from_utf8(output.stdout).unwrap().trim().to_string();

  //     // Construct the Merkle tree
  //     let tree = MerkleTree::construct_merkle_tree(git_root_path.clone()).await;
  //     // Check that the tree is not empty
  //     assert_ne!(tree.files.len(), 0);
  // 	println!("tree.files.len() = {}", tree.files.len());

  // 	// print the tree to a file.
  // 	let mut file = File::create("tree.txt").unwrap();
  // 	match write!(file, "{}", tree) {
  // 		Ok(_) => {},
  // 		Err(e) => println!("Failed to write to file: {}", e),
  // 	}
  // 	// write another file with the name of the files
  // 	let mut file = File::create("tree_files.txt").unwrap();
  // 	// create a list of files as a vec
  // 	let mut files: Vec<String> = Vec::new();
  // 	for (file_path, _) in tree.files.iter() {
  // 		files.push(file_path.clone());
  // 	}
  // 	// then sort the vector of files
  // 	files.sort();

  // 	for file_path in files.iter() {
  // 		match writeln!(file, "{}", file_path) {
  // 			Ok(_) => {},
  // 			Err(e) => println!("Failed to write to file: {}", e),
  // 		}
  // 	}

  //     // Check that the root path is correct
  //     assert_eq!(tree.root_path, git_root_path);
  //   }
}
