use anyhow::{Context, Result};
use git2::{
  Delta, DiffFindOptions, DiffOptions, ErrorCode, Oid, Reference, Repository,
};
use napi::bindgen_prelude::{FromNapiValue, ToNapiValue};
use napi_derive::napi;
use std::collections::HashMap;
use std::io::Read;
use std::panic::{self};
use std::sync::Mutex;
use std::time::{Duration, SystemTime, UNIX_EPOCH};

use serde_derive::Serialize;

#[napi]
pub struct GitClient {
  repo: Mutex<Repository>,
}

#[napi]
impl GitClient {
  // MARK: #[Napi] methods

  #[napi(constructor)]
  pub fn new(absolute_root_directory: String) -> Result<Self, napi::Error> {
    panic::catch_unwind(|| {
      let repo = Repository::open(&absolute_root_directory)
        .with_context(|| {
          format!("Failed to open repository at {}", absolute_root_directory)
        })
        .map_err(|e| napi::Error::from_reason(e.to_string()))?;
      Ok(GitClient {
        repo: Mutex::new(repo),
      })
    })
    .unwrap_or_else(|_| {
      Err(napi::Error::from_reason(
        "Panic occurred in new".to_string(),
      ))
    })
  }

  #[napi]
  pub async fn get_total_commit_count(&self) -> napi::Result<i32> {
    panic::catch_unwind(|| {
      self
        .get_total_commit_count_internal()
        .map_err(|e| napi::Error::from_reason(e.to_string()))
    })
    .unwrap_or_else(|_| {
      Err(napi::Error::from_reason(
        "Panic occurred in get_total_commit_count".to_string(),
      ))
    })
  }

  #[napi]
  pub async fn get_commit_verify_data(
    &self,
    commit: String,
  ) -> napi::Result<VerifyData> {
    panic::catch_unwind(|| {
      self
        .get_verify_data_internal(&commit)
        .map_err(|e| napi::Error::from_reason(e.to_string()))
    })
    .unwrap_or_else(|_| {
      Err(napi::Error::from_reason(
        "Panic occurred in get_commit_verify_data".to_string(),
      ))
    })
  }

  fn get_verify_data_internal(&self, commit: &str) -> Result<VerifyData> {
    let repo = &self
      .repo
      .lock()
      .map_err(|e| anyhow::anyhow!("Failed to lock repo: {}", e))?;

    let oid = Oid::from_str(commit).context("Invalid commit hash")?;
    let commit = repo.find_commit(oid).context("Failed to find commit")?;
    let commit_time = commit.time().seconds();

    let tree = commit.tree().context("Failed to get commit tree")?;
    let parent = commit.parent(0).ok();
    let parent_tree = parent.as_ref().and_then(|p| p.tree().ok());

    let diff =
      repo.diff_tree_to_tree(parent_tree.as_ref(), Some(&tree), None)?;

    let mut changed_files: Vec<String> = Vec::new();
    diff.foreach(
      &mut |delta, _| {
        if let Some(new_file) = delta.new_file().path() {
          if let Some(path) = new_file.to_str() {
            changed_files.push(path.to_string());
          }
        }
        true
      },
      None,
      None,
      None,
    )?;

    changed_files.sort();

    // Try to find a suitable blob from changed files
    for path in &changed_files {
      let entry = tree.get_path(std::path::Path::new(path)).ok();
      if let Some(entry) = entry {
        if let Ok(object) = entry.to_object(repo) {
          if let Some(blob) = object.as_blob() {
            if !blob.is_binary() && !blob.content().is_empty() {
              let content = String::from_utf8_lossy(
                &blob.content()[..std::cmp::min(1000, blob.content().len())],
              )
              .into_owned();
              return Ok(VerifyData {
                commit_time,
                file_content: content,
                file_path: path.clone(),
              });
            }
          }
        }
      }
    }

    // If no suitable blob found in changed files, search the entire tree
    let mut stack = vec![tree];
    while let Some(tree) = stack.pop() {
      for entry in tree.iter() {
        match entry.kind() {
          Some(git2::ObjectType::Blob) => {
            let object = entry.to_object(repo)?;
            if let Some(blob) = object.as_blob() {
              if !blob.is_binary() && !blob.content().is_empty() {
                let content = String::from_utf8_lossy(
                  &blob.content()[..std::cmp::min(1000, blob.content().len())],
                )
                .into_owned();
                let file_path = entry.name().unwrap_or("").to_string();
                return Ok(VerifyData {
                  commit_time,
                  file_content: content,
                  file_path,
                });
              }
            }
          }
          Some(git2::ObjectType::Tree) => {
            let object = entry.to_object(repo)?;
            if let Some(subtree) = object.as_tree() {
              stack.push(subtree.clone());
            }
          }
          _ => continue,
        }
      }
    }

    Err(anyhow::anyhow!("No suitable file found in the commit"))
  }

  #[napi]
  pub async fn throw_if_commit_doesnt_exist(
    &self,
    root_sha: String,
  ) -> napi::Result<()> {
    panic::catch_unwind(|| {
      self
        .throw_if_commit_doesnt_exist_internal(&root_sha)
        .map_err(|e| napi::Error::from_reason(e.to_string()))
    })
    .unwrap_or_else(|_| {
      Err(napi::Error::from_reason(
        "Panic occurred in throw_if_commit_doesnt_exist".to_string(),
      ))
    })
  }

  fn throw_if_commit_doesnt_exist_internal(
    &self,
    root_sha: &str,
  ) -> Result<()> {
    let repo = &self
      .repo
      .lock()
      .map_err(|e| anyhow::anyhow!("Failed to lock repo: {}", e))?;

    let oid = Oid::from_str(&root_sha).context("Invalid root SHA")?;
    let _ = repo.find_commit(oid).context("Failed to find commit")?;
    Ok(())
  }

  #[napi]
  pub async fn get_verify_commit(&self) -> napi::Result<String> {
    panic::catch_unwind(|| {
      self
        .get_verify_commit_internal()
        .map_err(|e| napi::Error::from_reason(e.to_string()))
    })
    .unwrap_or_else(|_| {
      Err(napi::Error::from_reason(
        "Panic occurred in get_verify_commit".to_string(),
      ))
    })
  }

  fn get_verify_commit_internal(&self) -> Result<String> {
    let repo = &self
      .repo
      .lock()
      .map_err(|e| anyhow::anyhow!("Failed to lock repo: {}", e))?;

    // Get the HEAD reference
    let head_ref = repo.head().context("Failed to get HEAD reference")?;
    let mut commit = head_ref
      .peel_to_commit()
      .context("Failed to peel to commit")?;

    // Calculate the timestamp for one month ago
    let one_month_ago = SystemTime::now()
      .checked_sub(Duration::from_secs(30 * 24 * 60 * 60))
      .ok_or_else(|| anyhow::anyhow!("Failed to calculate one month ago"))?;

    loop {
      // Get commit time as SystemTime
      let commit_time = UNIX_EPOCH
        .checked_add(Duration::from_secs(commit.time().seconds() as u64))
        .ok_or_else(|| anyhow::anyhow!("Invalid commit timestamp"))?;

      // Check if the commit is older than one month ago or has no parents
      if commit_time <= one_month_ago || commit.parent_count() == 0 {
        // Get parent tree if available
        let parent_tree = if commit.parent_count() > 0 {
          Some(commit.parent(0)?.tree()?)
        } else {
          None
        };

        // Get current commit tree
        let commit_tree = commit.tree()?;

        // Generate diff between parent tree and current commit tree
        let diff = repo.diff_tree_to_tree(
          parent_tree.as_ref(),
          Some(&commit_tree),
          None,
        )?;

        // Check if there are any changes
        if diff.deltas().len() > 0 || commit.parent_count() == 0 {
          return Ok(commit.id().to_string());
        }
      }

      // Move to the parent commit
      if commit.parent_count() > 0 {
        commit = commit.parent(0)?;
      } else {
        break;
      }
    }

    Err(anyhow::anyhow!("No suitable commit found"))
  }

  #[napi]
  pub async fn get_repo_head_sha(&self) -> napi::Result<Option<String>> {
    panic::catch_unwind(|| self.get_repo_head_sha_internal()).unwrap_or_else(
      |_| {
        Err(napi::Error::from_reason(
          "Panic occurred in get_repo_head_sha".to_string(),
        ))
      },
    )
  }

  fn get_repo_head_sha_internal(&self) -> napi::Result<Option<String>> {
    let repo = self
      .repo
      .lock()
      .map_err(|e| napi::Error::from_reason(e.to_string()))?;
    let head = repo.head();

    match head {
      Ok(head) => head
        .resolve()
        .and_then(|resolved_head: Reference| -> Result<String, git2::Error> {
          match resolved_head.target() {
            Some(oid) => Ok(oid.to_string()),
            None => Err(git2::Error::from_str("Failed to get root SHA")),
          }
        })
        .map(Some)
        .map_err(|e| {
          napi::Error::from_reason(format!("Failed to get root SHA: {}", e))
        }),
      Err(ref e)
        if e.code() == ErrorCode::UnbornBranch
          || e.code() == ErrorCode::NotFound =>
      {
        Ok(None)
      }
      Err(e) => Err(napi::Error::from_reason(format!(
        "Failed to get repository head: {}",
        e
      ))),
    }
  }

  #[napi]
  pub async fn get_commit_chain(
    &self,
    hash: String,
    depth: i32,
    get_files: CommitChainGetFiles,
  ) -> napi::Result<Vec<CommitData>> {
    panic::catch_unwind(|| {
      let mut commit_chain = Vec::new();
      let mut current_hash = hash;

      for _ in 0..depth {
        match self.get_full_commit_from_hash_internal(
          &current_hash,
          get_files == CommitChainGetFiles::DoGetFiles,
        ) {
          Ok(commit_data) => {
            commit_chain.push(commit_data);

            // now get the parent hash
            let parents = self.get_parent_shas_internal(&current_hash);
            match parents {
              Ok(parents) => {
                if let Some(parent_hash) = parents.first() {
                  current_hash = parent_hash.clone();
                } else {
                  break;
                }
              }
              Err(e) => {
                return Err(napi::Error::from_reason(format!(
                  "Failed to get parent hashes: {}",
                  e
                )));
              }
            }
          }
          Err(e) => {
            return Err(napi::Error::from_reason(format!(
              "Failed to get commit data: {}",
              e
            )))
          }
        }
      }

      Ok(commit_chain)
    })
    .unwrap_or_else(|_| {
      Err(napi::Error::from_reason(
        "Panic occurred in get_commit_chain".to_string(),
      ))
    })
  }

  // MARK: Internal methods

  fn get_parent_shas_internal(&self, hash: &str) -> Result<Vec<String>> {
    let oid = Oid::from_str(hash).context("Invalid hash")?;
    let repo = &self
      .repo
      .lock()
      .map_err(|e| anyhow::anyhow!("Failed to lock repo: {}", e))?;
    let commit = repo.find_commit(oid).context("Failed to find commit")?;

    Ok(commit.parent_ids().map(|oid| oid.to_string()).collect())
  }

  fn get_full_commit_from_hash_internal(
    &self,
    hash: &str,
    include_diff: bool,
  ) -> Result<CommitData> {
    let repo = &self
      .repo
      .lock()
      .map_err(|e| anyhow::anyhow!("Failed to lock repo: {}", e))?;

    let oid = Oid::from_str(hash).context("Invalid hash")?;
    let commit = repo.find_commit(oid).context("Failed to find commit")?;

    let mut commit_data = CommitData {
      sha: commit.id().to_string(),
      date: (commit.time().seconds() as f64) * 1000.0,
      message: commit.message().unwrap_or("").to_string(),
      author: commit.author().name().unwrap_or("").to_string(),
      parents: commit.parent_ids().map(|oid| oid.to_string()).collect(),
      files: Vec::new(),
    };

    if include_diff {
      let parent = commit.parent(0).ok();
      let parent_tree = parent.as_ref().and_then(|p| p.tree().ok());
      let commit_tree = commit.tree().context("Failed to get commit tree")?;

      let mut diff_options = DiffOptions::new();
      diff_options.ignore_submodules(true);

      let mut diff = repo.diff_tree_to_tree(
        parent_tree.as_ref(),
        Some(&commit_tree),
        Some(&mut diff_options),
      )?;

      let mut diff_find_options = DiffFindOptions::new();
      diff_find_options.exact_match_only(true);
      diff_find_options.rename_limit(100_000);

      diff.find_similar(Some(&mut diff_find_options))?;

      let mut file_additions_map = HashMap::<String, i32>::new();
      let mut file_deletions_map = HashMap::<String, i32>::new();

      diff.foreach(
        &mut |delta, _| {
          let old_file = delta.old_file();
          let new_file = delta.new_file();

          let old_path = old_file.path().and_then(|p| p.to_str()).unwrap_or("");
          let new_path = new_file.path().and_then(|p| p.to_str()).unwrap_or("");

          commit_data.files.push(CommitFile {
            from: old_path.to_string(),
            to: new_path.to_string(),
            additions: 0,
            deletions: 0,
            status: match delta.status() {
              Delta::Added => CommitFileStatus::Added,
              Delta::Deleted => CommitFileStatus::Deleted,
              Delta::Modified => CommitFileStatus::Modified,
              Delta::Renamed => CommitFileStatus::Renamed,
              _ => CommitFileStatus::Modified,
            },
          });
          true
        },
        None,
        None,
        Some(&mut |delta, _, line| {
          let new_file = delta.new_file();
          let new_path = new_file.path().and_then(|p| p.to_str()).unwrap_or("");

          let line_origin = line.origin();
          if line_origin == '+' {
            *file_additions_map.entry(new_path.to_string()).or_insert(0) += 1;
          } else if line_origin == '-' {
            *file_deletions_map.entry(new_path.to_string()).or_insert(0) += 1;
          }

          true
        }),
      )?;

      for file in commit_data.files.iter_mut() {
        let additions = file_additions_map.get(&file.to);
        let deletions = file_deletions_map.get(&file.to);

        if let Some(additions) = additions {
          file.additions = *additions;
        }

        if let Some(deletions) = deletions {
          file.deletions = *deletions;
        }
      }
    }

    Ok(commit_data)
  }

  fn get_total_commit_count_internal(&self) -> Result<i32> {
    let repo = self
      .repo
      .lock()
      .map_err(|e| anyhow::anyhow!("Failed to lock repo: {}", e))?;
    let mut revwalk = repo.revwalk().context("Failed to create revwalk")?;
    revwalk.push_head()?;
    Ok(revwalk.count() as i32)
  }
}

#[napi(object)]
#[derive(Debug, Serialize)]
pub struct CommitData {
  pub sha: String,
  pub date: f64,
  pub message: String,
  pub author: String,
  pub parents: Vec<String>,
  pub files: Vec<CommitFile>,
}

#[napi(object)]
#[derive(Debug, Serialize)]
pub struct CommitFile {
  pub from: String,
  pub to: String,
  pub additions: i32,
  pub deletions: i32,
  pub status: CommitFileStatus,
}

#[napi(object)]
#[derive(Debug, PartialEq, Eq, Hash, Serialize)]
pub enum CommitFileStatus {
  Added,
  Deleted,
  Modified,
  Renamed,
}

#[napi(object)]
#[derive(Debug, PartialEq, Eq, Hash, Serialize)]
pub enum CommitChainGetFiles {
  DoGetFiles,
  DontGetFiles,
}

#[napi(object)]
#[derive(Debug, Serialize)]
pub struct VerifyData {
  pub commit_time: i64,
  pub file_content: String,
  pub file_path: String,
}

#[cfg(test)]
mod tests {
  use super::*;
  use git2::{Signature, Time};
  use tempfile::TempDir;

  fn setup_test_repo() -> (TempDir, GitClient) {
    let temp_dir = TempDir::new().unwrap();
    let repo = Repository::init(temp_dir.path()).unwrap();

    // Create some commits
    let signature =
      Signature::new("Test User", "test@example.com", &Time::new(61, 0))
        .unwrap();

    // First commit
    let tree_id = repo.index().unwrap().write_tree().unwrap();
    let tree = repo.find_tree(tree_id).unwrap();
    repo
      .commit(
        Some("HEAD"),
        &signature,
        &signature,
        "Initial commit",
        &tree,
        &[],
      )
      .unwrap();

    // Second commit
    let tree_id = repo.index().unwrap().write_tree().unwrap();
    let tree = repo.find_tree(tree_id).unwrap();
    let parent = repo.head().unwrap().peel_to_commit().unwrap();
    repo
      .commit(
        Some("HEAD"),
        &signature,
        &signature,
        "Second commit",
        &tree,
        &[&parent],
      )
      .unwrap();

    let git_client =
      GitClient::new(temp_dir.path().to_str().unwrap().to_string()).unwrap();
    (temp_dir, git_client)
  }

  #[test]
  fn test_get_total_commit_count() {
    let (_temp_dir, git_client) = setup_test_repo();
    let count = git_client.get_total_commit_count_internal().unwrap();
    assert_eq!(count, 2, "Expected exactly two commits");
  }

  #[test]
  fn test_get_root_sha() {
    let (_temp_dir, git_client) = setup_test_repo();
    let root_sha = git_client.get_repo_head_sha_internal().unwrap();
    assert!(root_sha.is_some(), "Expected a non-empty root SHA");
  }

  #[test]
  #[ignore]
  fn test_get_full_commit_from_hash() {
    let repo_path = "/Users/sualeh/code/everysphere/portal-website";
    let git_client = GitClient::new(repo_path.to_string()).unwrap();
    let commit_data = git_client
      .get_full_commit_from_hash_internal(
        "bfa31fc6d7e7b34ff868ed5a04a77ccdf395ce73",
        true,
      )
      .unwrap();

    println!("{:?}", commit_data);

    // We don't know exactly how many files are in this commit, so we'll just check if it's greater than 0
    assert!(commit_data.files.len() == 3);

    // You might want to add more specific assertions based on what you know about this commit
    assert_eq!(commit_data.sha, "bfa31fc6d7e7b34ff868ed5a04a77ccdf395ce73");

    // Print out some information about the commit for verification
    println!("Commit message: {}", commit_data.message);
    println!("Author: {}", commit_data.author);
    println!("Number of files changed: {}", commit_data.files.len());

    // Optionally, print out details of the first few files
    for (i, file) in commit_data.files.iter().take(5).enumerate() {
      println!("File {}: {} -> {}", i + 1, file.from, file.to);
    }
  }

  #[test]
  #[ignore]
  fn test_get_verify_commit() {
    let repo_path = "/Users/arvid/code/everysphere";
    let git_client = GitClient::new(repo_path.to_string()).unwrap();
    let commit_data = git_client.get_verify_commit_internal().unwrap();
    println!("{:?}", commit_data);
    let verify_value =
      git_client.get_verify_data_internal(&commit_data).unwrap();
    println!("{:?}", verify_value);
  }

  #[test]
  #[ignore]
  fn test_get_verify_value() {
    let repo_path = "/Users/arvid/code/everysphere";
    let git_client = GitClient::new(repo_path.to_string()).unwrap();
    let verify_value = git_client
      .get_verify_data_internal("848da4fc214da3f2f365e08a42c4cdd830e47e83")
      .unwrap();
    println!("{:?}", verify_value);
  }

  #[test]
  #[ignore]
  fn test_throw_if_commit_doesnt_exist() {
    let repo_path = "/Users/arvid/code/everysphere";
    let git_client = GitClient::new(repo_path.to_string()).unwrap();
    git_client
      .throw_if_commit_doesnt_exist_internal(
        "7de7b9b9f8cd40d485b4d6ec2dd7a2b729662453",
      )
      .unwrap();
  }

  #[test]
  #[ignore]
  fn test_get_full_commit_from_hash_check_files_1() {
    let repo_path = "/Users/sualeh/code/everysphere/portal-website";
    let git_client = GitClient::new(repo_path.to_string()).unwrap();
    let commit_data = git_client
      .get_full_commit_from_hash_internal(
        "8bfe43b8b5ef57fadd1fbab1fce5c198b4202ef2",
        true,
      )
      .unwrap();

    // println!("{}", serde_json::to_string(&commit_data).unwrap());

    // We don't know exactly how many files are in this commit, so we'll just check if it's greater than 0
    // assert!(commit_data.files.len() == 3);

    // print list of froms
    for file in commit_data.files.iter() {
      println!("{}", file.to);
    }

    // You might want to add more specific assertions based on what you know about this commit
    assert_eq!(commit_data.sha, "8bfe43b8b5ef57fadd1fbab1fce5c198b4202ef2");

    // Print out some information about the commit for verification
    // println!("Commit message: {}", commit_data.message);
    // println!("Author: {}", commit_data.author);
    println!("Number of files changed: {}", commit_data.files.len());

    // Optionally, print out details of the first few files
    // for (i, file) in commit_data.files.iter().take(5).enumerate() {
    //   println!("File {}: {} -> {}", i + 1, file.from, file.to);
    // }
  }
}
