use crate::merkle_tree;
use log::{debug, error, info};
use std::{collections::HashSet, vec};

use napi_derive::napi;

#[napi]
pub struct MerkleClient {
  tree: merkle_tree::MerkleTree,
  absolute_root_directory: String,
}

use merkle_tree::{LocalConstruction, MerkleTree};

#[napi]
impl MerkleClient {
  #[napi(constructor)]
  pub fn new(absolute_root_directory: String) -> MerkleClient {
    // let canonical_root_directory = std::path::Path::new(&absolute_root_directory);
    // use dunce::canonicalize;
    // let canonical_root_directory = match dunce::canonicalize(&canonical_root_directory) {
    //   Ok(path) => path.to_str().unwrap_or(&absolute_root_directory).to_string().to_lowercase(),
    //   Err(e) => {
    //     info!("Error in canonicalizing path: path: {:?}, error {:?}", canonical_root_directory, e);
    //     absolute_root_directory
    //   }
    // };

    MerkleClient {
      tree: MerkleTree::empty_tree(),
      absolute_root_directory,
    }
  }

  #[napi]
  pub async fn is_too_big(
    &self,
    max_files: i32,
    git_ignored_files: Vec<String>,
    _is_git_repo: bool,
  ) -> bool {
    let git_ignored_set =
      HashSet::<String>::from_iter(git_ignored_files.into_iter());
    let mut num_files = 0;
    let mut dirs_to_check = vec![self.absolute_root_directory.clone()];

    while let Some(dir) = dirs_to_check.pop() {
      info!("dir: {:?}", dir);
      let mut entries = match tokio::fs::read_dir(&dir).await {
        Ok(entries) => entries,
        Err(_) => continue,
      };
      if num_files > max_files {
        return true;
      }

      while let Some(entry) = entries.next_entry().await.unwrap_or(None) {
        let path = entry.path();
        info!("entry: {:?}", path);
        let path_str = match path.to_str() {
          Some(path_str) => path_str.to_string(),
          None => continue,
        };

        if git_ignored_set.contains(&path_str) {
          continue;
        }

        match entry.file_type().await {
          Ok(file_type) => {
            if file_type.is_dir() {
              dirs_to_check.push(path_str);
            }

            if file_type.is_file() {
              num_files += 1;
            }
          }
          Err(_) => continue,
        }
      }
    }
    num_files > max_files
  }

  #[napi]
  pub async unsafe fn init(
    &mut self,
    git_ignored_files: Vec<String>,
    is_git_repo: bool,
  ) -> Result<(), napi::Error> {
    // 1. compute the merkle tree
    // 2. update the backend
    // 3. sync with the remote
    info!("Merkle tree compute started!");
    info!("Root directory: {:?}", self.absolute_root_directory);
    unsafe {
      self
        .compute_merkle_tree(git_ignored_files, is_git_repo)
        .await?;
    }

    Ok(())
  }

  #[napi]
  pub async unsafe fn init_with_ripgrep_ignore(
    &mut self,
  ) -> Result<(), napi::Error> {
    // 1. compute the merkle tree
    // 2. update the backend
    // 3. sync with the remote
    info!("Merkle tree compute with ripgrep ignore started!");
    info!("Root directory: {:?}", self.absolute_root_directory);
    unsafe {
      self.compute_merkle_tree_with_ripgrep_ignore().await?;
    }

    Ok(())
  }

  pub async unsafe fn interrupt(&mut self) -> Result<(), napi::Error> {
    unimplemented!("Interrupt is not implemented yet");
  }

  #[napi]
  pub async unsafe fn compute_merkle_tree_with_ripgrep_ignore(
    &mut self,
  ) -> Result<(), napi::Error> {
    let directory = self.absolute_root_directory.clone();

    let (tx, rx) = tokio::sync::oneshot::channel();

    tokio::task::spawn_blocking(move || {
      futures::executor::block_on(
        construct_merkle_tree_oneshot_with_ripgrep_ignore(directory, tx),
      );
    });

    let t = rx.await.map_err(|e| {
      napi::Error::new(napi::Status::Unknown, format!("{:?}", e))
    })?;

    match t {
      Ok(tree) => {
        self.tree = tree;
        Ok(())
      }
      Err(e) => {
        Err(napi::Error::new(napi::Status::Unknown, format!("{:?}", e)))
      }
    }
  }

  #[napi]
  pub async unsafe fn compute_merkle_tree(
    &mut self,
    git_ignored_files: Vec<String>,
    is_git_repo: bool,
  ) -> Result<(), napi::Error> {
    // make the git ignored files into a hash set
    let mut git_ignored_set = HashSet::from_iter(git_ignored_files.into_iter());

    // if the hashset itself contains the root directory, then we should remove it.
    // this is because the root directory is not a file, and we don't want to ignore it.
    if git_ignored_set.contains(&self.absolute_root_directory) {
      git_ignored_set.remove(&self.absolute_root_directory);
    }

    let directory = self.absolute_root_directory.clone();

    let (tx, rx) = tokio::sync::oneshot::channel();

    tokio::task::spawn_blocking(move || {
      futures::executor::block_on(construct_merkle_tree_oneshot(
        directory,
        git_ignored_set,
        is_git_repo,
        tx,
      ));
    });

    let t = rx.await.map_err(|e| {
      napi::Error::new(napi::Status::Unknown, format!("{:?}", e))
    })?;

    match t {
      Ok(tree) => {
        self.tree = tree;
        Ok(())
      }
      Err(e) => {
        Err(napi::Error::new(napi::Status::Unknown, format!("{:?}", e)))
      }
    }
  }

  #[napi]
  pub async unsafe fn update_file(&mut self, file_path: String) {
    let _ = self.tree.update_file(file_path).await;
  }

  #[napi]
  pub async unsafe fn delete_file(&mut self, file_path: String) {
    let _r = self.tree.delete_file(file_path);
  }

  #[napi]
  pub async fn get_subtree_hash(
    &self,
    relative_path: String,
  ) -> Result<String, napi::Error> {
    debug!("get_subtree_hash: relative_path: {:?}", relative_path);

    let relative_path_without_leading_slash = match relative_path
      .strip_prefix('.')
    {
      Some(path) => path.strip_prefix(std::path::MAIN_SEPARATOR).unwrap_or(""),
      None => relative_path.as_str(),
    };
    debug!(
      "relative_path_without_leading_slash: {:?}",
      relative_path_without_leading_slash
    );

    let absolute_path = if !relative_path_without_leading_slash.is_empty() {
      std::path::Path::new(&self.absolute_root_directory)
        .join(relative_path_without_leading_slash)
    } else {
      std::path::Path::new(&self.absolute_root_directory).to_path_buf()
    };

    debug!("absolute_path: {:?}", absolute_path);

    let absolute_path_string = match absolute_path.to_str() {
      Some(path) => path.to_string(),
      None => {
        return Err(napi::Error::new(
          napi::Status::Unknown,
          format!("some string error"),
        ))
      }
    };

    debug!("absolute_path_string: {:?}", absolute_path_string);

    let hash = self
      .tree
      .get_subtree_hash(absolute_path_string.as_str())
      .await;

    match hash {
      Ok(hash) => Ok(hash),
      Err(e) => Err(napi::Error::new(
        napi::Status::Unknown,
        format!("Error in get_subtree_hash. \nRelative path: {:?}, \nAbsolute path: {:?}, \nRoot directory: {:?}\nError: {:?}", &relative_path, absolute_path, self.absolute_root_directory, e)
      )),
    }
  }

  #[napi]
  pub async fn get_num_embeddable_files(&self) -> Result<i32, napi::Error> {
    let num = self.tree.get_num_embeddable_files().await;

    match num {
      Ok(num) => Ok(num),
      Err(e) => Err(napi::Error::new(
        napi::Status::Unknown,
        format!("Error in get_num_embeddable_files: {:?}", e),
      )),
    }
  }

  // return a list of "important" paths
  // this will include directories weighted by number of files, as well as some heuristics for important files (e.g. README, main.rs, lib.rs, etc)
  // the order of the paths is in order from most to least important
  #[napi]
  pub async fn get_important_paths(
    &self,
    k: i64,
  ) -> Result<Vec<String>, napi::Error> {
    let num = self.tree.get_important_paths(k).await;

    match num {
      Ok(num) => Ok(num),
      Err(e) => Err(napi::Error::new(
        napi::Status::Unknown,
        format!("Error in get_important_paths: {:?}", e),
      )),
    }
  }

  pub async fn get_num_embeddable_files_in_subtree(
    &self,
    relative_path: String,
  ) -> Result<i32, napi::Error> {
    let absolute_path = std::path::Path::new(&self.absolute_root_directory)
      .join(relative_path)
      .canonicalize()?;

    let num = self
      .tree
      .get_num_embeddable_files_in_subtree(absolute_path)
      .await;

    match num {
      Ok(num) => Ok(num),
      Err(e) => Err(napi::Error::new(
        napi::Status::Unknown,
        format!("Error in get_num_embeddable_files_in_subtree: {:?}", e),
      )),
    }
  }

  #[napi]
  pub async fn get_all_files(&self) -> Result<Vec<String>, napi::Error> {
    let files = self.tree.get_all_files().await;

    match files {
      Ok(files) => Ok(files),
      Err(e) => Err(napi::Error::new(
        napi::Status::Unknown,
        format!("Error in get_all_files: {:?}", e),
      )),
    }
  }

  #[napi]
  pub async fn get_all_dir_files_to_embed(
    &self,
    absolute_file_path: String,
  ) -> Result<Vec<String>, napi::Error> {
    // let absolute_path = absolute_file_path.to_lowercase();
    // let absolute_path_str = absolute_path.as_str();

    let files = self
      .tree
      .get_all_dir_files_to_embed(absolute_file_path.as_str())
      .await;

    match files {
      Ok(files) => Ok(files),
      Err(e) => Err(napi::Error::new(
        napi::Status::Unknown,
        format!("Error in get_all_dir_files_to_embed: {:?}", e),
      )),
    }
  }

  #[napi]
  pub async unsafe fn get_next_file_to_embed(
    &mut self,
  ) -> Result<Vec<String>, napi::Error> {
    let n = self.tree.get_next_file_to_embed().await;

    match n {
      Ok((file, path)) => {
        // now our job is to put the filename as the first element of the path.

        // TODO(sualeh): we should assert that the path is ascending up to the path.

        let ret = vec![file];
        let ret = ret.into_iter().chain(path.into_iter()).collect::<Vec<_>>();
        Ok(ret)
      }
      Err(e) => Err(napi::Error::new(
        napi::Status::Unknown,
        format!("Error in get_next_file_to_embed: {:?}", e),
      )),
    }
  }

  // FIXME(sualeh): get_spline
  #[napi]
  pub async fn get_spline(
    &self,
    absolute_file_path: String,
  ) -> Result<Vec<String>, napi::Error> {
    // let absolute_path = absolute_file_path.to_lowercase();
    // let absolute_path_str = absolute_path.as_str();
    let spline = self.tree.get_spline(absolute_file_path.as_str()).await;

    match spline {
      Ok(spline) => Ok(spline),
      Err(e) => Err(napi::Error::new(
        napi::Status::Unknown,
        format!("Error in get_spline: {:?}", e),
      )),
    }
  }

  #[napi]
  pub async fn get_hashes_for_files(
    &self,
    files: Vec<String>,
  ) -> Result<Vec<String>, napi::Error> {
    let hashes = self.tree.get_hashes_for_files(files).await;

    match hashes {
      Ok(hashes) => Ok(hashes),
      Err(e) => Err(napi::Error::new(
        napi::Status::Unknown,
        format!("Error in get_hashes_for_files: {:?}", e),
      )),
    }
  }

  #[napi]
  pub fn update_root_directory(&mut self, root_directory: String) {
    self.absolute_root_directory = root_directory;
  }
}

async fn construct_merkle_tree_oneshot(
  absolute_path_to_root_directory: String,
  git_ignored_files_and_dirs: HashSet<String>,
  is_git_repo: bool,
  tx: tokio::sync::oneshot::Sender<Result<MerkleTree, anyhow::Error>>,
) {
  let result = MerkleTree::construct_merkle_tree(
    absolute_path_to_root_directory,
    git_ignored_files_and_dirs,
    is_git_repo,
  )
  .await;
  let _ = tx.send(result);
}

async fn construct_merkle_tree_oneshot_with_ripgrep_ignore(
  absolute_path_to_root_directory: String,
  tx: tokio::sync::oneshot::Sender<Result<MerkleTree, anyhow::Error>>,
) {
  let result = MerkleTree::construct_merkle_tree_with_ripgrep_ignore(
    absolute_path_to_root_directory,
  )
  .await;
  let _ = tx.send(result);
}
