use crate::merkle_tree::{
  File, MerkleNode, MerkleNodePtr, NodeType, PinnedFuture,
};

use super::{LocalConstruction, MerkleTree};
use log::{debug, error, info};
use std::collections::{BTreeMap, HashMap, HashSet};
use std::path::{Path, PathBuf};
use tonic::async_trait;

// Create a max path const
const MAX_RELATIVE_PATH_LEN: usize = 250;

#[async_trait]
impl LocalConstruction for MerkleTree {
  async fn new(
    root_directory: Option<String>,
  ) -> Result<MerkleTree, anyhow::Error> {
    let git_ignored_files = HashSet::<String>::new();
    if let Some(root_directory) = root_directory {
      let n = MerkleTree::construct_merkle_tree(
        root_directory,
        git_ignored_files,
        false,
      )
      .await;
      return n;
    }

    Ok(MerkleTree::empty_tree())
  }

  async fn construct_merkle_tree_with_ripgrep_ignore(
    absolute_path_to_root_directory: String,
  ) -> Result<MerkleTree, anyhow::Error> {
    let path = PathBuf::from(absolute_path_to_root_directory.clone());
    if !path.exists() {
      // FIXME: we should report this via a good logger.
      return Err(anyhow::anyhow!("Root directory does not exist!"));
    }

    let git_ignored_files = HashSet::<String>::new();

    let walk_result = walk_dir(&path)?;

    let root_node = MerkleNode::construct_node_given_paths(
      &path,
      None,
      &git_ignored_files,
      &walk_result,
      absolute_path_to_root_directory.as_str(),
    )
    .await;

    let mut mt = MerkleTree {
      root: root_node,
      files: BTreeMap::new(),
      root_path: absolute_path_to_root_directory,
      cursor: None,
      git_ignored_files_and_dirs: git_ignored_files,
      is_git_repo: false,
    };

    // we now iterate over all the nodes and add them to the hashmap
    // TODO(later): i can make this parallel.
    fn add_nodes_to_hashmap<'a>(
      node: &'a MerkleNodePtr,
      files: &'a mut BTreeMap<String, File>,
    ) -> PinnedFuture<'a, ()> {
      Box::pin(async move {
        let node_reader = node.read().await;
        match &node_reader.node_type {
          NodeType::Branch(n) => {
            let children = &n.1;
            files.insert(n.0.clone(), File { node: node.clone() });
            for child in children {
              add_nodes_to_hashmap(child, files).await;
            }
          }
          NodeType::File(file_name) => {
            let f = File { node: node.clone() };

            // i dont reallly  like this :(((
            // let canonical_file_name = match dunce::canonicalize(file_name) {
            //   Ok(path) => path.to_str().unwrap_or(file_name).to_string(),
            //   Err(_) => file_name.clone(),
            // };

            files.insert(file_name.clone(), f);
          }
          NodeType::ErrorNode(_) => {
            // do nothing
          }
        }
      })
    }

    add_nodes_to_hashmap(&mt.root, &mut mt.files).await;

    info!(
      "number of files in the tree for rg ignore: {}",
      mt.files.len()
    );

    Ok(mt)
  }

  /// Factory method to construct a merkle tree from a given directory
  ///
  /// plan:
  /// 1. get all files in the directory
  /// 2. compute hash for each file
  /// 3. construct merkle tree
  /// 4. return merkle tree
  async fn construct_merkle_tree(
    absolute_path_to_root_directory: String,
    git_ignored_files_and_dirs: HashSet<String>,
    is_git_repo: bool,
  ) -> Result<MerkleTree, anyhow::Error> {
    let path = PathBuf::from(absolute_path_to_root_directory.clone());
    if !path.exists() {
      // FIXME: we should report this via a good logger.
      return Err(anyhow::anyhow!("Root directory does not exist!"));
    }

    // 1. get all the gitignored files
    // let git_ignored_files_and_dirs =
    //   match git_utils::list_ignored_files_and_directories(
    //     absolute_path_to_root_directory.as_str(),
    //     true,
    //   ) {
    //     Ok(git_ignored) => git_ignored,
    //     Err(_e) => HashSet::new(),
    //   };

    let root_node = MerkleNode::new(
      path,
      None,
      &git_ignored_files_and_dirs,
      absolute_path_to_root_directory.as_str(),
      is_git_repo,
    )
    .await;

    let mut mt = MerkleTree {
      root: root_node,
      files: BTreeMap::new(),
      root_path: absolute_path_to_root_directory,
      cursor: None,
      git_ignored_files_and_dirs,
      is_git_repo,
    };

    // we now iterate over all the nodes and add them to the hashmap
    // TODO(later): i can make this parallel.
    fn add_nodes_to_hashmap<'a>(
      node: &'a MerkleNodePtr,
      files: &'a mut BTreeMap<String, File>,
    ) -> PinnedFuture<'a, ()> {
      Box::pin(async move {
        let node_reader = node.read().await;
        match &node_reader.node_type {
          NodeType::Branch(n) => {
            let children = &n.1;
            files.insert(n.0.clone(), File { node: node.clone() });
            for child in children {
              add_nodes_to_hashmap(child, files).await;
            }
          }
          NodeType::File(file_name) => {
            let f = File { node: node.clone() };

            // i dont reallly  like this :(((
            // let canonical_file_name = match dunce::canonicalize(file_name) {
            //   Ok(path) => path.to_str().unwrap_or(file_name).to_string(),
            //   Err(_) => file_name.clone(),
            // };

            files.insert(file_name.clone(), f);
          }
          NodeType::ErrorNode(_) => {
            // do nothing
          }
        }
      })
    }

    add_nodes_to_hashmap(&mt.root, &mut mt.files).await;

    info!("number of files in the tree: {}", mt.files.len());

    Ok(mt)
  }

  async fn update_file(
    &mut self,
    file_path: String,
  ) -> Result<(), anyhow::Error> {
    // 1. check if the file exists.
    //   if it does, then update the file
    // 2. if it does not, then add the file

    let path = Path::new(&file_path);

    if !path.exists() {
      return Err(anyhow::anyhow!("File does not exist!"));
    }

    let path_exists_in_tree = self.files.contains_key(&file_path);

    if path_exists_in_tree {
      // Update the file in two places.
      let file = self.files.get(&file_path).unwrap();

      let r = self.update_node(file.node.clone()).await;

      if r.is_err() {
        // FIXME(sualeh): we should report this via a good logger.
        error!("Could not update node! {}", r.unwrap_err().to_string());
        return Err(anyhow::anyhow!("Could not update node!"));
      }
    } else {
      // File does not exist in the tree, let's add it.
      let e = self.attach_new_node_to_tree(file_path.clone()).await;

      if e.is_err() {
        return Err(anyhow::anyhow!("Could not attach new node to tree!"));
      }
    }

    Ok(())
  }

  async fn delete_file(
    &mut self,
    file_path: String,
  ) -> Result<(), anyhow::Error> {
    // 1. check if the file exists.
    //   if it does, then delete the file
    //
    // 2. if it does not, then do nothing

    let path = Path::new(&file_path);

    if !path.exists() {
      return Err(anyhow::anyhow!("File does not exist!"));
    }

    let path_exists_in_tree = self.files.contains_key(&file_path);

    if !path_exists_in_tree {
      return Err(anyhow::anyhow!("File does not exist in the tree!"));
    }

    // Get the parent node
    let parent_node = self.get_parent_node(&path);
    if parent_node.is_err() {
      return Err(anyhow::anyhow!("Could not get parent node! 0"));
    }
    let parent_node = parent_node.unwrap();
    let mut mut_parent = parent_node.write().await;

    // BUG(sualeh): need to actually drop everything that is a child here.
    // idea: enumerate all nodes that are children of this through a starts with query on the hashtable.
    // then drop all of them.
    // in opposite order of length.

    // then remove the node from the parent and you are done

    // Remove the child from the parent node
    match mut_parent.node_type {
      NodeType::Branch(ref mut node) => {
        let children = &mut node.1;
        let mut found = false;
        let mut index = 0;

        for (i, child) in children.iter().enumerate() {
          let child_reader = child.read().await;
          match &child_reader.node_type {
            NodeType::File(child_file_path) => {
              // TODO(sualeh): THERE HAS TO BE A BETTER WAY TO DO THIS.
              if child_file_path == &file_path {
                found = true;
                index = i;
                break;
              }
            }
            NodeType::Branch(_) => {
              continue;
            }
            NodeType::ErrorNode(_) => {
              continue;
            }
          }
        }

        if found {
          // remove the child from the parent.
          let _ = children.remove(index);
        }
      }
      NodeType::File(_) => {
        return Err(anyhow::anyhow!(
          "delete_file: This should not happen! parent was a file ..."
        ));
      }
      NodeType::ErrorNode(_) => {
        return Err(anyhow::anyhow!(
          "delete_file: This should not happen! parent was an error node ..."
        ));
      }
    };

    // need to propogate the change up the tree.
    mut_parent.update_hash_of_ancestor_path();

    // Remove the file from the files map
    self.files.remove(&file_path);

    Ok(())
  }
}

pub struct WalkResult {
  // these are all absolute paths
  pub children: HashMap<String, Vec<String>>,
}

// TODO: do a global ignore file just like we did back in the day? then users can override if they want which is nice
const BAD_DIRECTORIES: &[&str] = &[
  ".egg-info",
  ".venv",
  "node_modules",
  "__pycache__",
  ".next",
  ".nuxt",
  ".cache",
  ".sass-cache",
  ".gradle",
  ".DS_Store",
  ".ipynb_checkpoints",
  ".pytest_cache",
  ".mypy_cache",
  ".tox",
  ".git",
  ".hg",
  ".svn",
  ".bzr",
  ".lock-wscript",
  ".Python",
  ".jupyter",
  ".history",
  ".yarn",
  ".yarn-cache",
  ".eslintcache",
  ".parcel-cache",
  ".cache-loader",
  ".nyc_output",
  ".node_repl_history",
  ".pnp.js",
  ".pnp",
];

// may return an error if there are too many paths! for example
fn walk_dir(root: &Path) -> Result<WalkResult, anyhow::Error> {
  let mut walk_builder = ignore::WalkBuilder::new(root);

  let mut children = HashMap::<String, Vec<String>>::new();

  let mut file_counter = 0;

  let root_dir_length = root.to_str().unwrap_or("").len();
  let max_absolute_path_length = MAX_RELATIVE_PATH_LEN + root_dir_length;

  for result in walk_builder
    .hidden(false)
    .parents(true)
    .git_ignore(true)
    .ignore(true)
    .git_global(true)
    .git_exclude(true)
    .require_git(false)
    // Ignores all .git dirs
    .filter_entry(|entry| {
      let path = entry.path();
      !BAD_DIRECTORIES
        .iter()
        .any(|bad_dir| path.ends_with(bad_dir))
    })
    .max_filesize(Some(2 * 1024 * 1024))
    // maybe we should follow links? but it feels safest not to... probably, most links are in node_modules and things like that. also, it probably messes up some assumptions
    .follow_links(false)
    .add_custom_ignore_filename(".cursorignore")
    .build()
  {
    match result {
      Ok(entry) => {
        let path = entry.path();
        let path_str = path.to_str();
        if path == root {
          continue;
        }

        match path_str {
          Some(p) => {
            if p.len() > max_absolute_path_length {
              // Info log and continue
              debug!("Path too long: {}", p);
              continue;
            }
            file_counter += 1;
            if file_counter > 100_000 {
              return Err(anyhow::anyhow!("Too many files. Max # of files is 100,000. Please .cursorignore some files."));
            }
            if let Some(parent_path) = path.parent() {
              if let Some(parent_path_str) = parent_path.to_str() {
                let parent_path_string = parent_path_str.to_string();
                let child_path = p.to_string();

                if let Some(children) = children.get_mut(&parent_path_string) {
                  children.push(child_path);
                } else {
                  children.insert(parent_path_string, vec![child_path]);
                }
              }
            }
          }
          None => {
            debug!("Failed to convert path to string");
            continue;
          }
        }
      }
      Err(_) => {
        // do nothing
        // let's just debug log
        debug!("Failed to walk directory");
      }
    }
  }

  Ok(WalkResult { children })
}

#[tokio::test]
async fn test_walk_dir() {
  let path = Path::new("/Users/amansanger/Desktop/Side_Projects/kohls_src");
  let result = walk_dir(&path);
  assert!(
    result.is_ok(),
    "Failed to walk directory: {:?}",
    result.err().unwrap()
  );
  // Log the result
  let result = result.unwrap();
  for (parent, children) in result.children.iter() {
    println!("Parent: {}", parent);
    for child in children {
      println!("Child: {}", child);
    }
  }
}
