use super::file_utils;
use log::{debug, error, info};
use sha2::Digest;
use std::collections::BinaryHeap;
use std::collections::{BTreeMap, HashSet};
use std::path::PathBuf;
use std::vec;
use std::{fs, path::Path, sync::Arc};
use tokio::sync::RwLock;
use tonic::async_trait;

pub mod local_construction;
pub mod test;

pub type MerkleNodePtr = Arc<RwLock<MerkleNode>>;
// IMPORTANT: for recursion
// IMPORTANT: for recursion
// IMPORTANT: for recursion
// if you want to do recursion with async, please please use this type!!!
type PinnedFuture<'a, T> = Pin<Box<dyn Future<Output = T> + Send + 'a>>;
// or talk to sualeh to convince you.

struct MerkleNodePtrWrapper {
  pub weight: f64,
  pub node: MerkleNodePtr,
}

impl std::cmp::Ord for MerkleNodePtrWrapper {
  fn cmp(&self, other: &Self) -> std::cmp::Ordering {
    self
      .weight
      .partial_cmp(&other.weight)
      .unwrap_or(std::cmp::Ordering::Equal)
  }
}

impl std::cmp::PartialOrd for MerkleNodePtrWrapper {
  fn partial_cmp(&self, other: &Self) -> Option<std::cmp::Ordering> {
    Some(self.cmp(other))
  }
}

impl std::cmp::Eq for MerkleNodePtrWrapper {}

impl std::cmp::PartialEq for MerkleNodePtrWrapper {
  fn eq(&self, other: &Self) -> bool {
    self.weight == other.weight
  }
}

#[derive(Debug)]
pub struct MerkleTree {
  root_path: String,
  root: MerkleNodePtr,
  // files are probably all non-error nodes
  files: BTreeMap<String, File>,
  cursor: Option<usize>,
  git_ignored_files_and_dirs: HashSet<String>,
  is_git_repo: bool,
}

#[derive(Debug)]
pub struct File {
  node: MerkleNodePtr,
}

type FileName = String;
type ParentPtr = Option<MerkleNodePtr>;

#[derive(Debug)]
pub enum NodeType {
  Branch((FileName, Vec<MerkleNodePtr>)),
  // this is the file_name
  File(FileName),
  ErrorNode((FileName, String)),
}

#[derive(Debug)]
pub struct MerkleNode {
  pub id: i64,
  pub node_type: NodeType,
  pub hash: String,
  pub error: Option<String>,
  pub parent: ParentPtr,
  // the weight of the node is the number of files in the subtree rooted at this node.
  // if the node is a file, then the weight is 1.
  // the weights are lazily computed.
  pub weight: i64,
}

fn get_id() -> i64 {
  rand::random::<i64>()
}

// the merkle tree needs to implement 2 important traits
// 1. the local construction and update trait. this makes sure that it works well locally!
// 2. the remote sync trait. we need to make sure that it can be easily synced with the remote tree when necessary.

#[async_trait]
pub trait LocalConstruction {
  async fn new(
    root_directory: Option<String>,
  ) -> Result<MerkleTree, anyhow::Error>;

  async fn construct_merkle_tree(
    root_directory: String,
    git_ignored_files_and_dirs: HashSet<String>,
    is_git_repo: bool,
  ) -> Result<MerkleTree, anyhow::Error>;

  async fn construct_merkle_tree_with_ripgrep_ignore(
    root_directory: String,
  ) -> Result<MerkleTree, anyhow::Error>;

  async fn update_file(
    &mut self,
    file_path: String,
  ) -> Result<(), anyhow::Error>;

  async fn delete_file(
    &mut self,
    file_path: String,
  ) -> Result<(), anyhow::Error>;
}

// #[async_trait]
// pub trait RemoteSync {
//   async fn sync_with_remote(
//     &mut self,
//     client: super::RepositoryClient,
//   ) -> Result<Vec<File>, tonic::Status>;
//   async fn sync_subtree_node(
//     &mut self,
//     node: &MerkleNode,
//     client: super::RepositoryClient,
//   ) -> Result<Vec<File>, tonic::Status>;
// }

impl MerkleTree {
  pub fn empty_tree() -> MerkleTree {
    MerkleTree {
      root: Arc::new(RwLock::new(MerkleNode::empty_node(None, None))),
      files: BTreeMap::new(),
      root_path: "".to_string(),
      cursor: None,
      git_ignored_files_and_dirs: HashSet::new(),
      is_git_repo: false,
    }
  }

  pub async fn get_subtree_hash(
    &self,
    absolute_path: &str,
  ) -> Result<String, anyhow::Error> {
    debug!("get_subtree_hash: absolute_path: {:?}", absolute_path);

    let node = match self.files.get(absolute_path) {
      Some(file) => file.node.clone(),
      None => {
        let all_files: Vec<String> = self.files.keys().cloned().collect();
        return Err(anyhow::anyhow!(
          "Could not find file in tree! Looking for: {}. All files: {:?}",
          absolute_path,
          all_files
        ));
      }
    };

    let node_reader = node.read().await;
    let node_hash = node_reader.hash.clone();

    debug!("node_hash: {:?}", node_hash);

    Ok(node_hash)
  }

  pub async fn get_num_embeddable_files(&self) -> Result<i32, anyhow::Error> {
    let mut count = 0;

    for (_, file) in &self.files {
      let file_reader = file.node.read().await;
      match &file_reader.node_type {
        NodeType::File(_) => {
          count += 1;
        }
        NodeType::Branch(_) => {
          continue;
        }
        NodeType::ErrorNode(_) => {
          continue;
        }
      }
    }

    Ok(count)
  }

  async fn hydrate_weights(&self) {
    self.hydrate_weights_recursive(&self.root).await
  }

  fn hydrate_weights_recursive<'a>(
    &'a self,
    node: &'a MerkleNodePtr,
  ) -> PinnedFuture<'a, ()> {
    Box::pin(async move {
      let mut node_writer = node.write().await;
      match &node_writer.node_type {
        NodeType::File(_) => {
          node_writer.weight = 1;
        }
        NodeType::Branch((_, children)) => {
          let mut weight = 0;
          for child in children {
            self.hydrate_weights_recursive(child).await;
            let child_reader = child.read().await;
            weight += child_reader.weight;
          }
          node_writer.weight = weight;
        }
        NodeType::ErrorNode(_) => {
          node_writer.weight = 0;
        }
      }
    })
  }

  pub fn file_importance_factor(&self, path: String) -> f64 {
    let file_name = Path::new(&path).file_name();
    match file_name {
      Some(name) => {
        let name_str = name.to_str().unwrap_or("");
        match name_str.to_lowercase().as_str() {
          "readme.md" => 1.0,
          "package.json" | "cargo.toml" | "pom.xml" | "requirements.txt"
          | "setup.py" | "build.gradle" | "build.sbt" | "makefile" => 0.3,
          "main.ts" | "lib.rs" | "main.py" | "main.java" | "main.c"
          | "main.cpp" | "main.go" | "main.rb" | "main.php" | "main.swift"
          | "main.kt" | "main.hs" | "main.scala" | "main.pl" | "main.lua"
          | "main.r" | "main.jl" | "main.cs" | "main.f90" | "main.rs" => 0.1,
          _ => 0.0,
        }
      }
      None => 0.0,
    }
  }

  pub async fn get_important_paths(
    &self,
    k: i64,
  ) -> Result<Vec<String>, anyhow::Error> {
    self.hydrate_weights().await;

    // create a priority queue
    let mut queue: BinaryHeap<MerkleNodePtrWrapper> = BinaryHeap::new();
    queue.push(MerkleNodePtrWrapper {
      weight: self.root.read().await.weight as f64,
      node: Arc::clone(&self.root),
    });

    let mut paths = Vec::new();

    while let Some(x) = queue.pop() {
      let w = x.weight;
      let node = x.node;
      let node_reader = node.read().await;
      match &node_reader.node_type {
        NodeType::File(file_name) => {
          paths.push(file_name.clone());
        }
        NodeType::Branch((_, children)) => {
          for child in children {
            let child_reader = child.read().await;
            let weight = child_reader.weight as f64;
            let mut new_weight = weight;
            if let NodeType::File(file_name) = &child_reader.node_type {
              new_weight += self.file_importance_factor(file_name.clone()) * w;
            }
            queue.push(MerkleNodePtrWrapper {
              weight: new_weight,
              node: Arc::clone(child),
            });
          }
        }
        _ => {}
      }
      if paths.len() >= k as usize {
        break;
      }
    }

    Ok(paths)
  }

  pub async fn get_num_embeddable_files_in_subtree(
    &self,
    absolute_path: PathBuf,
  ) -> Result<i32, anyhow::Error> {
    let mut count = 0;

    let absolute_path = match absolute_path.to_str() {
      Some(s) => s.to_string(),
      None => {
        return Err(anyhow::anyhow!(
        "get_num_embeddable_files_in_subtree: Failed to convert path to string"
      ))
      }
    };

    // TODO(sualeh): worth keeping this list sorted. its now a btree

    for (_, file) in &self.files {
      let file_reader = file.node.read().await;
      match &file_reader.node_type {
        NodeType::File(file_name) => {
          if file_name.contains(&absolute_path) {
            count += 1;
          }
        }
        NodeType::Branch(_) => {
          continue;
        }
        NodeType::ErrorNode(_) => {
          continue;
        }
      }
    }

    Ok(count)
  }

  pub async fn get_all_files(&self) -> Result<Vec<String>, anyhow::Error> {
    let mut files = Vec::new();

    for (file_name, file) in &self.files {
      let file_reader = file.node.read().await;
      match &file_reader.node_type {
        NodeType::File(_) => {
          files.push(file_name.clone());
        }
        NodeType::Branch(_) => {
          continue;
        }
        NodeType::ErrorNode(_) => {
          continue;
        }
      }
    }

    Ok(files)
  }

  pub async fn get_hashes_for_files(
    &self,
    files: Vec<String>,
  ) -> Result<Vec<String>, anyhow::Error> {
    let mut hashes = Vec::new();

    for file_name in files {
      let file = match self.files.get(&file_name) {
        Some(file) => file,
        None => {
          return Err(anyhow::anyhow!("Could not find file in tree!"));
        }
      };

      let file_reader = file.node.read().await;
      match &file_reader.node_type {
        NodeType::File(_) => {
          hashes.push(file_reader.hash.clone());
        }
        NodeType::Branch(_) => {
          continue;
        }
        NodeType::ErrorNode(_) => {
          continue;
        }
      }
    }

    Ok(hashes)
  }

  /// Returns a filename, and then a path from its parent to the root (which can possibly be empty.)
  pub async fn get_next_file_to_embed(
    &mut self,
  ) -> Result<(String, Vec<String>), anyhow::Error> {
    // if the cursor is none, set it to 0
    let cursor = match self.cursor {
      Some(cursor) => cursor,
      None => {
        self.cursor = Some(0);
        0
      }
    };

    // get the thing at the cursor. while we dont find a file, we keep incrementing the cursor.
    let mut cursor = cursor;
    loop {
      // O(log n)
      let file = match self.files.values().nth(cursor) {
        Some(file) => file,
        None => {
          return Err(anyhow::anyhow!("Could not find file to embed!"));
        }
      };

      let file_reader = file.node.read().await;
      match &file_reader.node_type {
        NodeType::File(f) => {
          // update the cursor.
          self.cursor = Some(cursor + 1);
          let spline = self.get_spline(f).await?;
          return Ok((f.clone(), spline));
        }
        NodeType::Branch(_) => {
          cursor += 1;
          continue;
        }
        NodeType::ErrorNode(_) => {
          cursor += 1;
          continue;
        }
      }
    }
  }

  pub async fn get_all_dir_files_to_embed(
    &self,
    absolute_path: &str,
  ) -> Result<Vec<String>, anyhow::Error> {
    let mut files = Vec::new();

    // 1. should check that this absolute path is actually a directory.
    let file_node = self.files.get(absolute_path);
    if file_node.is_none() {
      return Err(anyhow::anyhow!("Could not find directory the in tree!"));
    }
    let absolute_path = if absolute_path.ends_with(std::path::MAIN_SEPARATOR) {
      absolute_path.to_string()
    } else {
      format!("{}{}", absolute_path, std::path::MAIN_SEPARATOR)
    };

    for (file_path, f) in &self.files {
      if !file_path.contains(absolute_path.as_str()) {
        continue;
      }

      match f.node.read().await.node_type {
        NodeType::File(_) => {
          files.push(file_path.clone());
        }
        NodeType::Branch(_) => {
          continue;
        }
        NodeType::ErrorNode(_) => {
          continue;
        }
      }
    }

    Ok(files)
  }

  // TODO(sualeh): i need tests for this!!
  pub async fn get_spline(
    &self,
    absolute_path: &str,
  ) -> Result<Vec<String>, anyhow::Error> {
    let mut files = Vec::new();

    let current_node = match self.files.get(absolute_path) {
      Some(node) => node.node.clone(),
      None => {
        return Err(anyhow::anyhow!("File not found: {}", absolute_path));
      }
    };

    let mut stack = Vec::new();
    stack.push(current_node);

    while let Some(node) = stack.pop() {
      let parent = node.read().await.parent.clone();
      if let Some(parent) = parent {
        {
          let parent_node = parent.read().await;
          match &parent_node.node_type {
            NodeType::File(file_name) => {
              files.push(file_name.clone());
            }
            NodeType::Branch((branch_name, _)) => {
              files.push(branch_name.clone());
            }
            _ => {
              continue;
            }
          }
        }

        stack.push(parent);
      }
    }
    Ok(files)
  }

  /// creates a new node and attaches it to the current tree.
  /// SPEC:
  /// - you are allowed to create a file with a node such that the
  ///   file is contained within the subtree defined by the root.
  /// - updates the hash of the ancestor path.
  /// - attaches to the ancestor.
  /// - doenst add to the filemap
  /// Primary usecase:
  /// 1. when a new file is created.
  /// Don't use it for:
  /// 1. creating a full new tree.
  /// Returns:
  /// - the new node that was created.
  async fn create_new_node_and_attach_to_ancestors(
    &mut self,
    file_path: PathBuf,
  ) -> Result<MerkleNodePtr, anyhow::Error> {
    // algorithm:
    // 1. find the nearest parent before we reach the root.
    // 2. if we don't find a parent, we should definitely return an error.
    // 3. then we should create the chain of things down. then attach it to the parent we have.
    let (ancestor, path) = match self.get_ancestor_in_tree(&file_path) {
      Ok((ancestor, path)) => (ancestor, path),
      Err(e) => {
        return Err(anyhow::anyhow!(
          "Could not find ancestor in tree! {}",
          e.to_string()
        ));
      }
    };

    // 2 cases:
    // 1. the path is empty. this means that the ancestor is the root.
    // 2. the path is non-empty. that means there exist a non-empty element btwn till the root.

    let absolute_root_path = self.root_path.clone();
    let new_node = match path.len() {
      0 => {
        // this means that the ancestor is the root.
        // we need to create a new node and attach it to the ancestor.
        let new_node = MerkleNode::new(
          file_path.clone(),
          Some(ancestor.clone()),
          &self.git_ignored_files_and_dirs,
          &absolute_root_path.as_str(),
          self.is_git_repo,
        )
        .await;
        ancestor.write().await.attach_child(new_node.clone()).await;
        new_node
      }
      _ => {
        // this means that the ancestor is not the root.
        // we need to create a new node and attach it to the ancestor.

        // UNSURE: not sure this is the correct thing to do but it is the fastest.
        // get the last thing that is not in the tree.
        // UNWRAP CHECKED AND FINE
        let first_child_path = path.last().unwrap();
        let first_child = MerkleNode::new(
          first_child_path.clone(),
          Some(ancestor.clone()),
          &self.git_ignored_files_and_dirs,
          &absolute_root_path.as_str(),
          self.is_git_repo,
        )
        .await;

        // TODO(sualeh): we should do an assertion check that the entire vec is contained here.

        // now we need to attach the first child to the ancestor.
        let mut mut_ancestor = ancestor.write().await;
        mut_ancestor.attach_child(first_child.clone()).await;
        first_child
      }
    };

    Ok(new_node)
  }

  /// Spec:
  /// - adds to the tree but doesnt change things which already exist in the file.
  async fn add_subtree_to_filemap(&mut self, node: MerkleNodePtr) {
    // Traverse the subtree rooted at 'node' and add all the nodes to the file map
    let mut stack = Vec::new();
    stack.push(node);

    while !stack.is_empty() {
      // UNWRAP CHECKED AND FINE
      let current_node = match stack.pop() {
        Some(node) => node,
        None => continue,
      };

      let node = current_node.read().await;
      match &node.node_type {
        NodeType::File(file_path) => {
          // SUALEH BE CAREFUL!
          match self.files.get(file_path) {
            Some(file) => {
              // check if the hash is the same:
              let file_hash = file.node.read().await.hash.clone();

              if file_hash != node.hash {
                // update the file from the map.
                self.files.insert(
                  file_path.clone(),
                  File {
                    node: current_node.clone(),
                  },
                );
              }
            }
            None => {
              self.files.insert(
                file_path.clone(),
                File {
                  node: current_node.clone(),
                },
              );
            }
          }
        }
        NodeType::Branch(node) => {
          let children = &node.1;
          for child in children {
            stack.push(child.clone());
          }
        }
        NodeType::ErrorNode(_) => {
          continue;
        }
      }
    }
  }

  /// PRECONDITION: the file exists in the tree!!
  /// SPEC:
  /// - attaches to the ancestor.
  /// - adds to the filemap
  /// - updates hashes of ancestor path.
  async fn attach_new_node_to_tree(
    &mut self,
    file_path: String,
  ) -> Result<(), anyhow::Error> {
    let path = PathBuf::from(file_path.clone());
    match self.create_new_node_and_attach_to_ancestors(path).await {
      Ok(node_ptr) => {
        self.add_subtree_to_filemap(node_ptr).await;
        Ok(())
      }
      Err(e) => Err(anyhow::anyhow!(
        "Could not create new node and attach to ancestors! {}",
        e.to_string()
      )),
    }
  }

  /// MUTATES MUTATES MUTATES
  /// Precondition:
  /// - this can only be called on a file node!!!
  async fn update_node(
    &mut self,
    node: MerkleNodePtr,
  ) -> Result<(), anyhow::Error> {
    let node_reader = node.read().await;

    // first
    let file_string = match &node_reader.node_type {
      NodeType::File(file_path) => file_path,
      NodeType::Branch(_) => {
        return Err(anyhow::anyhow!(
          "Unexpected NodeType::Branch encountered in method 'update_node'."
        ));
      }
      NodeType::ErrorNode(_) => {
        return Err(anyhow::anyhow!(
          "Unexpected NodeType::ErrorNode encountered in method 'update_node'."
        ));
      }
    };
    let file_string = file_string.clone();
    let file_path = PathBuf::from(file_string.clone());

    debug!("file_string: {:?}", file_string);
    // get the parent, and remove this node from the parent.
    let parent_node = node_reader.parent.clone();

    match parent_node {
      Some(parent_node) => {
        let mut mut_parent = parent_node.write().await;

        // check if the parent node has the children i want.
        match mut_parent.node_type {
          NodeType::Branch(ref mut node) => {
            let children = &mut node.1;
            let mut found = false;
            let mut index = 0;

            for (i, child) in children.iter().enumerate() {
              match &child.read().await.node_type {
                NodeType::File(child_file_path) => {
                  if child_file_path == &file_string {
                    found = true;
                    index = i;
                    break;
                  }
                }
                NodeType::Branch(_) => {
                  continue;
                }
                NodeType::ErrorNode(_) => {
                  continue;
                }
              }
            }

            if found {
              // remove the child from the parent.
              let _ = children.remove(index);
            }
          }
          NodeType::File(_) => {
            return Err(anyhow::anyhow!(
              "Unexpected NodeType::File encountered in parent node."
            ));
          }
          NodeType::ErrorNode(_) => {
            return Err(anyhow::anyhow!(
              "Unexpected NodeType::ErrorNode encountered in parent node."
            ));
          }
        };
      }
      None => {
        return Err(anyhow::anyhow!(
          "Parent node is missing. This should not happen."
        ));
      }
    }

    let node_ptr = self
      .create_new_node_and_attach_to_ancestors(file_path)
      .await;

    match node_ptr {
      Ok(node_ptr) => {
        self.files.insert(
          file_string,
          File {
            node: node_ptr.clone(),
          },
        );
      }
      Err(e) => {
        return Err(anyhow::anyhow!(
          "Could not create new node and attach to ancestors! {}",
          e.to_string()
        ));
      }
    }

    Ok(())
  }

  fn get_parent_node<'a>(
    &self,
    file_path: &'a Path,
  ) -> Result<MerkleNodePtr, anyhow::Error> {
    let parent_name = file_path.parent();

    match parent_name {
      Some(parent_name) => {
        debug!("parent_name: {:?}", parent_name);
        let parent_name = match parent_name.to_str() {
          Some(name) => name.to_string(),
          None => {
            return Err(anyhow::anyhow!(
              "Failed to convert parent_name to string"
            ))
          }
        };
        let parent_node = match self.files.get(&parent_name) {
          Some(file) => file.node.clone(),
          None => {
            return Err(anyhow::anyhow!("Could not get parent node! 2"));
          }
        };

        Ok(parent_node)
      }
      None => {
        return Err(anyhow::anyhow!("Could not get parent node! 3"));
      }
    }
  }

  /// SPEC:
  /// This function retrieves the ancestor node in the Merkle tree for a given file path.
  /// It returns a tuple containing the ancestor node and a vector of paths from the ancestor to the file.
  /// If the file is not in the tree, an error is returned.
  ///
  /// Postcondition:
  /// - the vector contains all the paths not including the ancestor itself.
  ///
  fn get_ancestor_in_tree<'a>(
    &self,
    file_path: &'a Path,
  ) -> Result<(MerkleNodePtr, Vec<PathBuf>), anyhow::Error> {
    let mut current_path = file_path;
    let mut path_vec = Vec::new();
    let root = self.root_path.clone();

    match current_path.to_str() {
      Some(path_str) => {
        if !path_str.contains(&root) {
          return Err(anyhow::anyhow!("File is not in the tree!"));
        }
      }
      None => {
        return Err(anyhow::anyhow!("Could not convert path to string!"));
      }
    }

    loop {
      let parent_path = current_path.parent();

      match parent_path {
        Some(parent_path) => match parent_path.to_str() {
          Some(parent_name) => {
            if self.files.contains_key(parent_name) {
              match self.files.get(parent_name) {
                Some(file) => return Ok((file.node.clone(), path_vec)),
                None => {
                  return Err(anyhow::anyhow!(
                    "Wow, this is weird. This should not happen!"
                  ));
                }
              };
            } else {
              path_vec.push(parent_path.to_path_buf());
              current_path = parent_path;
            }
          }
          None => {
            return Err(anyhow::anyhow!("Could not convert path to string!"));
          }
        },
        None => {
          return Err(anyhow::anyhow!("Couldn't find ancestor node!"));
        }
      };

      // check if we are at the root.
      match current_path.to_str() {
        Some(path_str) => {
          if path_str == &root {
            return Err(anyhow::anyhow!("Unexpected error: Reached the root of the tree while searching for ancestor node."));
          }
        }
        None => {
          return Err(anyhow::anyhow!("Could not convert path to string!"));
        }
      }
    }
  }
}

use std::future::Future;
use std::pin::Pin;

type IgnoredFiles = HashSet<String>;

impl MerkleNode {
  /// please be careful using this.
  async fn __new_unchecked(
    file_or_directory: String,
    parent: ParentPtr,
    ignored_files: &IgnoredFiles,
    absolute_root_path: &str,
    is_git_repo: bool,
  ) -> MerkleNodePtr {
    // // check if the root is a git directory.
    // let is_git_repo =
    //   match git_utils::is_git_directory(absolute_root_path).await {
    //     Ok(is_git_repo) => is_git_repo,
    //     Err(_e) => false,
    //   };
    let bypass_git = !is_git_repo;

    MerkleNode::construct_node(
      Path::new(&file_or_directory),
      parent,
      ignored_files,
      absolute_root_path,
      bypass_git,
    )
    .await
  }

  // #[tracing::instrument]
  async fn new(
    absolute_file_or_directory: PathBuf,
    parent: ParentPtr,
    ignored_files: &IgnoredFiles,
    absolute_root_path: &str,
    is_git_repo: bool,
  ) -> MerkleNodePtr {
    let bypass_git = !is_git_repo;

    info!(
      "constructing node for absolute_file_or_directory: {:?}",
      absolute_file_or_directory
    );

    MerkleNode::construct_node(
      Path::new(&absolute_file_or_directory),
      parent,
      ignored_files,
      absolute_root_path,
      bypass_git,
    )
    .await
  }

  /// NOT added to the tree by default.
  // async fn from_file(file_path: String) -> MerkleNode {
  // 	let path = Path::new(&file_path);
  // 	if path.exists() {
  // 		// Update the file in two places.
  // 		let file_hash = self.files.get_mut(&file_path).unwrap();

  fn construct_node_given_paths<'a>(
    absolute_file_or_directory: &'a Path,
    parent: ParentPtr,
    // hacky but this should always be empty
    ignored_files: &'a IgnoredFiles,
    walk_result: &'a local_construction::WalkResult,
    absolute_root_path: &'a str,
  ) -> PinnedFuture<'a, MerkleNodePtr> {
    Box::pin(async move {
      // check if it is a file
      let path_str = match absolute_file_or_directory.to_str() {
        Some(s) => s.to_string(),
        None => {
          error!("BAD PATH STR");
          "".to_string()
        }
      };

      if absolute_file_or_directory.is_file() {
        return Arc::new(RwLock::new(
          MerkleNode::construct_file_node_or_error_node(
            absolute_file_or_directory,
            parent,
            ignored_files,
          )
          .await,
        ));
      }

      let empty_paths = vec![];
      let children_paths =
        walk_result.children.get(&path_str).unwrap_or(&empty_paths);

      let mut children = Vec::<MerkleNodePtr>::new();

      let node = Arc::new(RwLock::new(MerkleNode {
        id: get_id(),
        node_type: NodeType::Branch((path_str.clone(), vec![])),
        hash: "".to_string(),
        error: None,
        parent,
        weight: 0,
      }));

      for children_path in children_paths {
        let entry = PathBuf::from(children_path.clone());
        match entry.exists() {
          true => {
            children.push(
              MerkleNode::construct_node_given_paths(
                &entry,
                Some(node.clone()),
                ignored_files,
                walk_result,
                absolute_root_path,
              )
              .await,
            );
          }
          false => {
            error!("error reading directory");
            // this looks kinda sketchy ngl....
            children.push(Arc::new(RwLock::new(MerkleNode::empty_node(
              Some(absolute_file_or_directory),
              Some("failed to read directory".to_string()),
            ))));
          }
        }
      }

      let node_hash = MerkleNode::compute_branch_hash(&children).await;
      // get a write lock on the node
      let mut node_writer = node.write().await;
      node_writer.node_type = NodeType::Branch((path_str.clone(), children));
      node_writer.hash = node_hash.clone();
      // release the lock
      drop(node_writer);

      node
    })
  }

  fn construct_node<'a>(
    absolute_file_or_directory: &'a Path,
    parent: ParentPtr,
    ignored_files: &'a IgnoredFiles,
    absolute_root_path: &'a str,
    bypass_git: bool,
  ) -> PinnedFuture<'a, MerkleNodePtr> {
    Box::pin(async move {
      // check if it is a file
      let path_str = match absolute_file_or_directory.to_str() {
        Some(s) => s.to_string(),
        None => {
          error!("BAD PATH STR");
          "".to_string()
        }
      };

      if absolute_file_or_directory.is_file() {
        return Arc::new(RwLock::new(
          MerkleNode::construct_file_node_or_error_node(
            absolute_file_or_directory,
            parent,
            ignored_files,
          )
          .await,
        ));
      }

      // check if the directory fails the bad dir test.
      let is_bad_dir = file_utils::is_in_bad_dir(absolute_file_or_directory);
      if is_bad_dir.is_err() || is_bad_dir.unwrap_or(false) {
        // println!("skipping directory: {}", path_str);
        return Arc::new(RwLock::new(MerkleNode::empty_node(
          Some(absolute_file_or_directory),
          Some("Directory is in bad dir!".to_string()),
        )));
      }

      let is_git_ignored_dir = ignored_files.contains(&path_str);

      if is_git_ignored_dir && !bypass_git {
        info!("skipping directory: {}", path_str);
        return Arc::new(RwLock::new(MerkleNode::empty_node(
          Some(absolute_file_or_directory),
          Some("Directory is git ignored!".to_string()),
        )));
      }

      let entries = fs::read_dir(absolute_file_or_directory);
      match entries {
        Ok(_) => (),
        Err(e) => {
          error!("error reading directory: {}", e);
          return Arc::new(RwLock::new(MerkleNode::empty_node(
            Some(absolute_file_or_directory),
            Some(e.to_string()),
          )));
        }
      }
      // UNWRAP CHECKED AND FINE
      let entries = entries.unwrap();
      let mut children = Vec::<MerkleNodePtr>::new();

      let node = Arc::new(RwLock::new(MerkleNode {
        id: get_id(),
        node_type: NodeType::Branch((path_str.clone(), vec![])),
        hash: "".to_string(),
        error: None,
        parent,
        weight: 0,
      }));

      for entry in entries {
        match entry {
          Ok(entry) => {
            children.push(
              MerkleNode::construct_node(
                &entry.path(),
                Some(node.clone()),
                ignored_files,
                absolute_root_path,
                bypass_git,
              )
              .await,
            );
          }
          Err(e) => {
            error!("error reading directory: {}", e);
            children.push(Arc::new(RwLock::new(MerkleNode::empty_node(
              Some(absolute_file_or_directory),
              Some(e.to_string()),
            ))));
          }
        }
      }

      let node_hash = MerkleNode::compute_branch_hash(&children).await;
      // get a write lock on the node
      let mut node_writer = node.write().await;
      node_writer.node_type = NodeType::Branch((path_str.clone(), children));
      node_writer.hash = node_hash.clone();
      // release the lock
      drop(node_writer);

      node
    })
  }

  async fn construct_file_node(
    absolute_file_path: &Path,
    parent: ParentPtr,
    ignored_files: &IgnoredFiles,
  ) -> Result<MerkleNode, String> {
    let file_str = absolute_file_path
      .to_str()
      .ok_or("Could not convert file path to string!")?
      .to_string();
    // first see if it passes the
    match file_utils::is_good_file(absolute_file_path) {
      Ok(_) => {}
      Err(e) => {
        return Err(format!("File failed runtime checks! {}", e.to_string()));
      }
    }

    // check if the file is in the git ignore buffer.
    // this is a bug right because we are not checking absoluteness here.
    match ignored_files.contains(&file_str) {
      true => {
        return Err(format!("File is in git ignore buffer!"));
      }
      false => {}
    }

    // check if the file passes runtime checks.
    match file_utils::is_good_file_runtime_check(
      absolute_file_path,
      // &file_content,
    )
    .await
    {
      Ok(_) => {}
      Err(e) => {
        return Err(format!("File failed runtime checks! {}", e.to_string()));
      }
    }

    // read the file_content to a buffer
    let file_content =
      match file_utils::read_string_without_bom(absolute_file_path).await {
        Ok(content) => content,
        Err(e) => {
          return Err(format!("Could not read file! {}", e.to_string()));
        }
      };

    let file_hash = compute_hash(&file_content);
    let node = MerkleNode {
      id: get_id(),
      node_type: NodeType::File(file_str),
      hash: file_hash.clone(),
      error: None,
      parent,
      weight: 0,
    };

    Ok(node)
  }

  async fn construct_file_node_or_error_node(
    absolute_file_path: &Path,
    parent: ParentPtr,
    ignored_files: &IgnoredFiles,
  ) -> MerkleNode {
    let node = match MerkleNode::construct_file_node(
      absolute_file_path,
      parent,
      ignored_files,
    )
    .await
    {
      Ok(node) => node,
      Err(e) => MerkleNode::empty_node(Some(absolute_file_path), Some(e)),
    };

    node
  }

  /// PRECONDITION: this must be a branch node.
  async fn attach_child(&mut self, child: MerkleNodePtr) {
    match &mut self.node_type {
      NodeType::Branch(node) => {
        let children = &mut node.1;
        children.push(child);
        self.update_hash_of_ancestor_path();
      }
      NodeType::File(_) => {
        error!(
          "Unexpected NodeType::File encountered while attaching child node."
        );
      }
      NodeType::ErrorNode(_) => {
        error!("Unexpected NodeType::ErrorNode encountered while attaching child node.");
      }
    }
  }

  async fn compute_branch_hash(children: &[MerkleNodePtr]) -> String {
    let mut hasher = sha2::Sha256::new();
    let mut names_and_hashes = vec![];
    let mut non_zero_children = 0;

    for child in children {
      // check if it is an error node
      let child_reader = child.read().await;

      match &child_reader.node_type {
        NodeType::File(file_name) => {
          non_zero_children += 1;
          names_and_hashes.push((file_name.clone(), child_reader.hash.clone()));
        }
        NodeType::Branch((file_name, _)) => {
          let hash = child_reader.hash.clone();
          if hash == "" {
            continue;
          }

          non_zero_children += 1;
          names_and_hashes.push((file_name.clone(), hash));
        }
        NodeType::ErrorNode(_) => {
          continue;
        }
      }
    }

    // sort the list of names and hashes by the hashes!!
    names_and_hashes
      .sort_by(|a, b| a.1.to_lowercase().cmp(&b.1.to_lowercase()));

    for (_name, hash) in names_and_hashes {
      if hash == "" {
        continue;
      }
      hasher.update(hash);
    }

    if non_zero_children == 0 {
      // this means that the branch is empty.
      // we should return an empty string.
      return "".to_string();
    }

    let result = hasher.finalize();
    format!("{:x}", result)
  }

  fn update_hash_of_ancestor_path<'a>(&'a mut self) -> PinnedFuture<'a, ()> {
    Box::pin(async move {
      self.hash = match &self.node_type {
        NodeType::Branch(node) => {
          let children = &node.1;
          MerkleNode::compute_branch_hash(children).await
        }
        NodeType::File(_) => {
          // TODO: this is kind of sketchy
          error!("Unexpected NodeType::File encountered in method 'update_hash_of_ancestor_path'.");
          "dummy_hash".to_string()
        }
        NodeType::ErrorNode(error) => {
          // TODO: this is kind of sketchy
          error!("Unexpected NodeType::ErrorNode encountered in method 'update_hash_of_ancestor_path'. Error: {:?}", error);
          "dummy_hash".to_string()
        }
      };

      // propogate it up to the parents as long as they are not null
      match &self.parent {
        Some(parent) => {
          let mut mut_parent = parent.write().await;
          mut_parent.update_hash_of_ancestor_path().await;
        }
        None => {
          error!("This node has no parent!");
        }
      }
    })
  }

  fn empty_node(file_name: Option<&Path>, error: Option<String>) -> MerkleNode {
    if let Some(file_name) = file_name {
      let file_name = file_name.to_str();
      match file_name {
        Some(file_name) => {
          return MerkleNode {
            id: get_id(),
            node_type: NodeType::ErrorNode((
              file_name.to_string(),
              error.clone().unwrap_or_default(),
            )),
            hash: "".to_string(),
            error,
            parent: None,
            weight: 0,
          };
        }
        None => {
          return MerkleNode {
            id: get_id(),
            node_type: NodeType::ErrorNode((
              "".to_string(),
              error.clone().unwrap_or_default(),
            )),
            hash: "".to_string(),
            error: Some(
              "FilenameError: Could not convert file name to string!"
                .to_string()
                + &error.unwrap_or_default(),
            ),
            parent: None,
            weight: 0,
          };
        }
      }
    } else {
      return MerkleNode {
        id: -1,
        node_type: NodeType::ErrorNode((
          "".to_string(),
          error.clone().unwrap_or_default(),
        )),
        hash: "".to_string(),
        error,
        parent: None,
        weight: 0,
      };
    }
  }

  fn fmt(
    &self,
    f: &mut std::fmt::Formatter<'_>,
    indent: usize,
  ) -> std::fmt::Result {
    match &self.node_type {
      NodeType::Branch(node) => {
        // print yourself
        writeln!(f, "{}BRANCH", " ".repeat(indent * 2))?;
        writeln!(
          f,
          "{}Branch: {}, Hash: {}",
          " ".repeat(indent * 2),
          node.0,
          self.hash
        )?;

        let children = node.1.clone();

        let names = children
          .iter()
          .map(|child| {
            let child_read = child.try_read();
            if let Ok(child) = child_read {
              match &child.node_type {
                NodeType::File(file_name) => file_name.clone(),
                NodeType::Branch((branch_name, _)) => branch_name.clone(),
                NodeType::ErrorNode(_) => "".to_string(),
              }
            } else {
              "".to_string()
            }
          })
          .collect::<Vec<String>>();

        let mut zipped_child_name: Vec<(&MerkleNodePtr, String)> =
          children.iter().zip(names).collect();

        zipped_child_name.sort_by(|a, b| a.1.cmp(&b.1));

        for child in &zipped_child_name {
          let child_read = child.0.try_read();
          if let Ok(child) = child_read {
            child.fmt(f, indent + 1)?;
          } else {
            writeln!(f, "------")?;
            writeln!(f, "Could not get read lock on a child node!")?;
            writeln!(f, "------")?;
          }
        }
      }
      NodeType::File(file_name) => {
        writeln!(f, "{}FILE", " ".repeat(indent * 2))?;
        if let Some(error) = &self.error {
          writeln!(
            f,
            "{}File: {}, Hash: {} Error: {}",
            " ".repeat(indent * 2),
            file_name,
            self.hash,
            error
          )?;
        } else {
          writeln!(
            f,
            "{}File: {}, Hash: {}",
            " ".repeat(indent * 2),
            file_name,
            self.hash,
          )?;
        }
      }
      NodeType::ErrorNode(_error) => {
        // writeln!(f, "{}ERROR:", " ".repeat(indent * 2))?;
        // writeln!(f, "{}file: {}", " ".repeat(indent * 2), error.0)?;
        // writeln!(f, "{}error: {}", " ".repeat(indent * 2), error.1)?;
      }
    }
    Ok(())
  }
}

fn compute_hash(file_content: &str) -> String {
  let mut hasher = sha2::Sha256::new();
  hasher.update(file_content);
  let result = hasher.finalize();

  format!("{:x}", result)
}

impl std::fmt::Display for MerkleTree {
  fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
    let root = self.root.try_read();
    if root.is_err() {
      write!(f, "Could not get read lock on root node!")?;
      write!(f, "the root path is: {}", self.root_path)?;
      write!(f, "---------------------------------\n")?;
      return Ok(());
    }

    // UNWRAP CHECKED AND FINE
    let root = root.unwrap();
    write!(f, "{}", root)
    // dont uncomment: this will be an infinite loop
    // write!(f, "Files: {:#?}\n", self.files)?;
    // write!(f, "Number of files: {}\n", self.files.len())?;
    // write!(f, "---------------------------------\n")
  }
}

impl std::fmt::Display for MerkleNode {
  fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
    self.fmt(f, 0)
  }
}

impl std::fmt::Display for NodeType {
  fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
    match self {
      NodeType::Branch(_) => write!(f, "Branch"),
      NodeType::File(file_name) => write!(f, "File: {}", file_name),
      NodeType::ErrorNode((file_name, error)) => {
        write!(f, "ErrorNode: {} {}", file_name, error)
      }
    }
  }
}
