var {
  _optionalChain
} = require('@sentry/utils');

Object.defineProperty(exports, '__esModule', { value: true });

const instrumentationIoredis = require('@opentelemetry/instrumentation-ioredis');
const instrumentationRedis4 = require('@opentelemetry/instrumentation-redis-4');
const core = require('@sentry/core');
const utils = require('@sentry/utils');
const instrument = require('../../otel/instrument.js');
const redisCache = require('../../utils/redisCache.js');

const INTEGRATION_NAME = 'Redis';

let _redisOptions = {};

const cacheResponseHook = (span, redisCommand, cmdArgs, response) => {
  span.setAttribute(core.SEMANTIC_ATTRIBUTE_SENTRY_ORIGIN, 'auto.db.otel.redis');

  const safeKey = redisCache.getCacheKeySafely(redisCommand, cmdArgs);
  const cacheOperation = redisCache.getCacheOperation(redisCommand);

  if (
    !safeKey ||
    !cacheOperation ||
    !_optionalChain([_redisOptions, 'optionalAccess', _ => _.cachePrefixes]) ||
    !redisCache.shouldConsiderForCache(redisCommand, safeKey, _redisOptions.cachePrefixes)
  ) {
    // not relevant for cache
    return;
  }

  // otel/ioredis seems to be using the old standard, as there was a change to those params: https://github.com/open-telemetry/opentelemetry-specification/issues/3199
  // We are using params based on the docs: https://opentelemetry.io/docs/specs/semconv/attributes-registry/network/
  const networkPeerAddress = _optionalChain([core.spanToJSON, 'call', _2 => _2(span), 'access', _3 => _3.data, 'optionalAccess', _4 => _4['net.peer.name']]);
  const networkPeerPort = _optionalChain([core.spanToJSON, 'call', _5 => _5(span), 'access', _6 => _6.data, 'optionalAccess', _7 => _7['net.peer.port']]);
  if (networkPeerPort && networkPeerAddress) {
    span.setAttributes({ 'network.peer.address': networkPeerAddress, 'network.peer.port': networkPeerPort });
  }

  const cacheItemSize = redisCache.calculateCacheItemSize(response);

  if (cacheItemSize) {
    span.setAttribute(core.SEMANTIC_ATTRIBUTE_CACHE_ITEM_SIZE, cacheItemSize);
  }

  if (redisCache.isInCommands(redisCache.GET_COMMANDS, redisCommand) && cacheItemSize !== undefined) {
    span.setAttribute(core.SEMANTIC_ATTRIBUTE_CACHE_HIT, cacheItemSize > 0);
  }

  span.setAttributes({
    [core.SEMANTIC_ATTRIBUTE_SENTRY_OP]: cacheOperation,
    [core.SEMANTIC_ATTRIBUTE_CACHE_KEY]: safeKey,
  });

  const spanDescription = safeKey.join(', ');

  span.updateName(utils.truncate(spanDescription, 1024));
};

const instrumentIORedis = instrument.generateInstrumentOnce('IORedis', () => {
  return new instrumentationIoredis.IORedisInstrumentation({
    responseHook: cacheResponseHook,
  });
});

const instrumentRedis4 = instrument.generateInstrumentOnce('Redis-4', () => {
  return new instrumentationRedis4.RedisInstrumentation({
    responseHook: cacheResponseHook,
  });
});

/** To be able to preload all Redis OTel instrumentations with just one ID ("Redis"), all the instrumentations are generated in this one function  */
const instrumentRedis = Object.assign(
  () => {
    instrumentIORedis();
    instrumentRedis4();

    // todo: implement them gradually
    // new LegacyRedisInstrumentation({}),
  },
  { id: INTEGRATION_NAME },
);

const _redisIntegration = ((options = {}) => {
  return {
    name: INTEGRATION_NAME,
    setupOnce() {
      _redisOptions = options;
      instrumentRedis();
    },
  };
}) ;

/**
 * Redis integration for "ioredis"
 *
 * Capture tracing data for redis and ioredis.
 */
const redisIntegration = core.defineIntegration(_redisIntegration);

exports.instrumentRedis = instrumentRedis;
exports.redisIntegration = redisIntegration;
